   #define autosleep                   ; Comment out this line if you want the backlight to stay on permanently, or if the display has no backlight.
                                       ; If autosleep is not defined, the backlight will remain at the backlightOnDutyCycle brightness.

;************************************************************
;  Ambient thermometer and thermostat
;************************************************************
;
;    Written by Andrew Partridge
;
;    This is version 1.0. Last modified on 14 October 2012
;
;    Author's email: andrewpartridge@tpg.com.au
;
;    This software is provided without warranty of any kind. It has not been validated for any application in which
;    property damage, personal injury or death may occur if it does not work as expected.
;
;    This software is free for personal use.
;
;    Please contact the author to discuss licensing for commercial applications, to report bugs, or to suggest improvements.
;
;************************************************************
;    Acknowledgements:
;
;       Uses fixed point maths routines for Microchip Technology Inc's 8-bit PIC processors, written by Frank J. Testa and
;       described in Microchip App Note AN617 (available on the Microchip website, www.microchip.com)
;
;       Uses code from the Silicon Chip May 2008 LC Meter firmware, LCmeter.asm, developed by Jim Rowe, for initialising the
;       LCD and converting binary to BCD.
;
;       Circuitry for PWM control of the display contrast has been copied from the Silicon Chip March 2010 
;       S/PDIF & TOSLink Digital Audio Signal Generator, by Nicholas Vinen. The PWM coding was written by the present author.
;
;       Uses code from Dallas Semiconductor for interfacing to the 1-wire DS18B20. The present author has modified this
;       code to operate with a 32MHz clock.
;
;************************************************************
;    General description:
;
;    This digital thermostat is based on a PIC16F1827 with a DS18B20 temperature sensor.
;
;    It features:
;    *  C or F display;
;    *  a range of -55C to +125C (-67F to 257F);
;    *  current temperature display resolution of 0.1 degree;
;    *  temperature accuracy of 0.5C for temperatures between -10C and +85C;
;    *  the switching temperature and hysteresis are adjustable in steps of 1 degree of the displayed temperature unit;
;    *  the hysteresis ranges from 0 to 180C (0 to 324F);
;    *  Auto (thermostatic) and Manual switching of the controlled output;
;    *  in Auto mode the controlled output can be normal or inverted;
;    *  adjustable LCD contrast; and
;    *  automatic backlight.
;
;    All settings are adjusted using a single pushbutton switch in conjunction with prompts from the display.
;   
;    Main display
;   
;    Suppose the user has set the mode to Auto, the switching temperature to 25C, the hysteresis to 2C, and the
;    output to be not inverted. The main display at power up will then look something like this:
;   
;    on>25 off<23
;    21.2C OFF
;   
;    The first line shows that the controlled output will be energised when the temperature exceeds 25C, and 
;    then stay on until the temperature drops below 23C. The second line is updated every second, and shows the 
;    current temperature, the temperature units, and the actual state of the controlled output.
;   
;    If the output is set to be inverted, the display would read instead:
;   
;    off>25 on<23
;    21.2C ON
;   
;    If the mode is set to Manual the main display is:
;   
;    Manual
;    21.2C OFF
;   
;    Setting adjustment
;   
;    A single momentary pushbutton switch, S1, is used for adjustment of user settings. When the main display is 
;    showing, a brief press of S1 enters the setting menu sequence. Menus allow the user to view and change the 
;    thermostat settings, the display contrast, and the displayed temperature units. 
;   
;    Briefly pressing S1 while in any of the setting menus skips the menu without making any changes, while 
;    pressing and holding S1 for longer than 1 second changes the setting. 
;   
;    There are two menus for each numerical setting: one which allows the setting to be increased, and one which 
;    allows the setting to be decreased.
;   
;    Numerical settings increase or decrease at an accelerating rate while S1 is held down, enabling rapid 
;    adjustment. The speed of adjustment is indicated by the number of '-' or '+' symbols displayed.
;   
;    If the user changes a numerical setting in one direction then briefly presses S1, the menu for adjusting 
;    that setting in the opposite direction is displayed. This allows any overshoot to be conveniently corrected.
;    A second brief press of S1 advances to the menu for the next setting.
;   
;    Settings are stored in EEPROM so that the user's most recent selections are restored the next time the unit 
;    is powered up.
;   
;    The display contrast is reset to maximum if the unit is powered up with button S1 already pressed. This 
;    allows the user to regain control in case the contrast has been set so faint that the display is impossible 
;    to read.
;   
;    A handy shortcut is available while the main display is showing: if the unit is in Manual mode a long press 
;    of S1 will toggle the state of the controlled output, while if the unit is in Auto mode a long press of S1 
;    will switch from Auto to Manual mode.
;   
;    In Auto mode, the controlled output will continue to be thermostatically controlled regardless of whether 
;    the main display or a setting menu is being shown.
;   
;    Automatic backlight
;   
;    The LCD backlight behaves in a similar fashion to a mobile phone backlight. It stays on at full brightness 
;    for 20 seconds after power-up and after S1 was last pressed, then is dimmed to 25% brightness for 10 seconds
;    before turning off. Unlike a mobile phone, the purpose of the automatic backlight is not to save power. 
;    Rather, if the unit is used in a bedroom having the backlight switch off allows the room to become 
;    completely dark. 
;   
;    Whenever the backlight is dimmed or off the first press of S1 restores it to full brightness and is not 
;    otherwise treated as a command, with one exception: the long press shortcuts from the main display operate 
;    regardless of the state of the backlight.
;   
;    If you use a display that has no backlight, or if it has a backlight but you want it to stay on permanently,
;    comment out the "#define autosleep" line in the assembly source file.
;   
;    Switching temperature units
;   
;    The thermostat switching temperature and hysteresis are always displayed as whole numbers of degrees, 
;    whether C or F. They are stored in units of 1/16C regardless of the temperature unit selected for 
;    display. Whenever the display unit is changed, the stored values of the thermostat temperature and 
;    hysteresis are adjusted slightly so that they are consistent with the displayed values.
    
;    For example, if the switching temperature is displaying as 12C (equivalent to 53.6F), the stored value is 
;    exactly 12.000C. If the display units are changed to Fahrenheit, the whole number Fahrenheit equivalent to 
;    12C is 54F. Since 54F is equivalent to 12.222C, the stored switching temperature is changed to 12.250C,
;    which is within 1/16C of the correct value. If the display units are then changed back to Celsius, the 
;    stored value of the switching temperature changes back to 12.000C.
;   
;    Circuit notes
;   
;    The 1N5817 schottky diode protects against reverse polarity of the 12V supply.
;   
;    The display backlight and relay are powered from the 11.6V rail. This keeps the current drawn from the 5V 
;    rail low enough that the 5V regulator will not need a heatsink. 
;   
;    The signal at RB6 of the PIC is pulse-width modulated to control the brightness of the backlight via a BC337
;    transistor. Full brightness corresponds to (almost) 100% duty cycle, and the duty cycle is reduced to 25% 
;    when it is required to dim the display, and then to 0% when the backlight is switched off.
;   
;    The value of the 150 ohm current limiting resistor in the backlight circuit was calculated for the display I
;    used, which has a backlight that is designed for direct connection to 5V and draws 44mA. If your display's 
;    backlight is designed for a different current or does not have its own current limiting resistors in series 
;    with the backlight LEDs you may need to change the value of the 150 ohm resistor to suit. Ensure you use a 
;    resistor with an appropriate power rating, as it will probably need to dissipate more than 1/4W.
;   
;    The display contrast is controlled by a variable current sink formed from a BC337 transistor driven by a PWM
;    signal from RA4. The duty cycle of the PWM corresponds to the contrast setting. A 1.5k resistor and 100nF
;    capacitor form a low-pass filter that averages the current over the PWM cycle.
;
;    The circuit is designed for a 12V relay with a 160 ohm coil. Choose a relay with contacts to suit the load 
;    you want to control.

;************************************************************
;  CPU configuration
;
   list P=PIC16F1827
   #include <P16F1827.INC>

;  _FOSC_INTOSC                        ; INTOSC Oscillator, I/O function on RA7/OSC1/CLKI
;  _WDTE_OFF                           ; WDT disabled
;  _PWRTE_ON                           ; PWRT enabled
;  _MCLRE_OFF                          ; RA5/MCLR/VPP pin function is digital input
;  _CP_OFF                             ; Program memory code protection is disabled
;  _CPD_OFF                            ; Data memory code protection is disabled
;  _BOREN_ON                           ; Brown-out Reset enabled
;  _CLKOUTEN_OFF                       ; CLKOUT function is disabled. I/O or oscillator function on RA6/CLKOUT
;  _IESO_OFF                           ; Internal/External Switchover mode is disabled
;  _FCMEN_OFF                          ; Fail-Safe Clock Monitor is disabled

;  _WRT_ALL                            ; 000h to FFFh write protected, no addresses may be modified by EECON control
;  _VCAPEN_OFF                         ; VCAP pin function is disabled
;  _PLLEN_OFF                          ; 4x PLL disabled
;  _STVREN_OFF                         ; Stack Overflow or underflow will not cause a Reset
;  _BORV_27                            ; Brown-out Reset Voltage (VBOR) set to 2.7 V
;  _LVP_OFF                            ; High voltage on MCLR/VPP must be used for programming

   __CONFIG    _CONFIG1, _FOSC_INTOSC & _WDTE_OFF & _PWRTE_ON & _MCLRE_OFF & _CP_OFF & _CPD_OFF & _BOREN_ON & _CLKOUTEN_OFF & _IESO_OFF & _FCMEN_OFF

   __CONFIG    _CONFIG2, _WRT_ALL & _VCAPEN_OFF & _PLLEN_OFF & _STVREN_ON & _BORV_27 & _LVP_OFF

;************************************************************
;  I/O Assignments
;
; RB0 - unused (output)
; RB1 - LCD2 (DB6)
; RB2 - thermostat output to relay driver
; RB3 - LCD0 (DB4)
; RB4 - LCD1 (DB5)
; RB5 - unused (output)
; RB6 - programming clock / P2A PWM control signal for display backlight
; RB7 - programming data / LCD3 (DB7)

; RA0 - unused (output)
; RA1 - LCD E (Ena)
; RA2 - unused (output)
; RA3 - unused (output)
; RA4 - contrast PWM out
; RA5 - multifunction button (connect switch to ground)
; RA6 - DS18B20 data
; RA7 - LCD RS

#define  LCD3                 PORTB,7  ; LCD module DB7
#define  LCD2                 PORTB,1  ; LCD module DB6
#define  LCD1                 PORTB,4  ; LCD module DB5
#define  LCD0                 PORTB,3  ; LCD module DB4

#define  ENA                  PORTA,1  ; LCD module 'Enable' control line
#define  RS                   PORTA,7  ; LCD module 'RS' control line

#define  thermostatOutput     PORTB,2  ; thermostat output to relay driver
#define  DQPORT               PORTA    ; Which port DS18B20 data line is on
#define  DQBIT                6        ; Which port bit DS18B20 data line is
#define  DQPRTBIT        DQPORT,DQBIT  ; The actual DS18B20 data line
#define  DQTRIS          TRISA,DQBIT   ; Direction bit for DS18B20 data line

#define  multifunctionButton  PORTA,5  ; (Digital Input) momentary SPST pushbutton switch to ground.
                                       ;    Port input has internal weak pullup resistor.
                                       ;    switch pressed = 0; switch not pressed = 1

;************************************************************
;  STATUS bit definitions

#define  _C   STATUS,C                 ; Carry bit
#define  _DC  STATUS,DC                ; Inter-nibble carry bit
#define  _Z   STATUS,Z                 ; Zero bit

;************************************************************
;  Backlight duty cycle constants
;
;  duty cycle = CCPR2L:CCP2CON<5:4> / ( 4 x (PR6 + 1) )
;             = CCPR2L / (PR6 + 1), since CCP2CON<5:4> = 00
;             = CCPR2L / h'100', since PR6 = h'FF'

   constant backlightOnDutyCycle = h'FF'  ; 99.6% duty cycle for the backlight when fully on
   constant backlightDimDutyCycle = h'40' ; 25% duty cycle for the backlight when dim
   constant backlightOffDutyCycle = h'00' ; 0% duty cycle for the backlight when off

;************************************************************
;  Backlight automatic dimming and switch-off constants

   constant backlightOnTime = d'30'    ; Total time (in seconds) before backlight switches off if button not pressed
   constant backlightDimTime = d'10'   ; Time (in seconds) that backlight is dimmed to backlightDimDutyCycle before turning off

;************************************************************
;  File register declarations for bank 4 registers (h'220' to h'24F')

;  Use the bank4indirect macro to enable indirect (INDF0) access to these locations.
;  Use the bank0indirect macro to revert to bank 0 and bank 1 indirect (INDF0) access.

   cblock h'220'
   endc

   cblock
   backlightTimer                      ; Counts seconds since the button was pressed, for the purpose of dimming and then turning off the backlight
   endc

;************************************************************
;  File register declarations for bank 0 registers (h'20' to h'6F')

   cblock h'20'                        ;  Bank 0 file registers begin at h'20'
   endc

;************************************************************
;  Variables used to maintain the thermostat settings
;  (thermostatInvert and thermostatManual are in the flags section)
;
;  These are all read from EEPROM during initialisation, and written back to EEPROM whenever they are changed

   cblock
   thermostatTemperatureL              ; 16 bit thermostat switching temperature (low byte)
   thermostatTemperatureH              ; 16 bit thermostat switching temperature (high byte)
   thermostatHysteresisL               ; 16 bit thermostat switching hysteresis in 1/16 degrees C (low byte)
   thermostatHysteresisH               ; 16 bit thermostat switching hysteresis in 1/16 degrees C (high byte)
   endc

#define thermostatTemperature thermostatTemperatureL
#define thermostatHysteresis thermostatHysteresisL

;************************************************************
;  Variables used for the accelerating-rate single-button change of numerical settings
;  (settingUp and changedSetting are in the flags section)

   cblock
;  settingUp                           ; true iff the setting value is being increased
;  changedSetting                      ; true iff the setting value has been changed (requiring it to be written back to EEPROM)
   endc
   cblock
   setting:2                           ; 16 bits (only 8 bits used for contrast adjustment). The value of the setting
   settingDelta:2                      ; 16 bits (only 8 bits used for contrast adjustment).
                                       ;   Amount by which the setting is due to change on the next increment or decrement
   endc
   cblock
   changesSkipped                      ; 8 bits. count of number of short presses of the multifunction button.
                                       ;    Adjustment of the setting stops when this reaches 2
   stage                               ; the low 4 bits (the high 4 bits are stageCounter). An index that records how fast the setting is being changed
;stageCounter equ stage                ; the high 4 bits of the shared register.
                                       ;    Counts from stage down to 0 to set the number of accelerator characters shown in settingMenu()
   changeCount                         ; 8 bits. count how many changes to the value of the setting since stage was last incremented
   endc

   cblock
   settingNameOffset                   ; 8 bits. Offset into the string table of the text preceding the setting value,
                                       ;    i.e. either "Switch at" or "Hysteresis".
   minSetting:2                        ; 16 bits. minimum value of the setting
   maxSetting:2                        ; 16 bits. maximum value of the setting
   endc

   cblock
   flags1                              ; General flags - see #define section for the use of each bit
   flags2                              ; General flags - see #define section for the use of each bit
   flags4                              ; General flags - see #define section for the use of each bit
   endc

   cblock
   currentDS18B20Temp:2                ; The most recent temperature sensor reading;
                                       ;   currentDS18B20Temp has the LSB, currentDS18B20Temp+1 has the MSB
   endc

;************************************************************
;  Locations used by the fixed point arithmetic routines

   cblock
   AARGB3                              ; 32-bit AARG LSB
   AARGB2                              ; 24-bit AARG LSB
   AARGB1                              ; 16-bit AARG LSB
   AARGB0                              ; 8-bit AARG (MSB)

   BARGB2                              ; 24-bit BARG LSB
   BARGB1                              ; 16-bit BARG LSB
   BARGB0                              ; 8-bit BARG (MSB)

   REMB1                               ; 16-bit remainder LSB
   REMB0                               ; 8-bit remainder (MSB)

   TEMPB3                              ; 32-bit temporary LSB
   TEMPB2                              ; 24-bit temporary LSB
   TEMPB1                              ; 16-bit temporary LSB
   TEMPB0                              ; 8-bit temporary (MSB)
   endc

TEMP equ TEMPB0
NAN  equ 4                             ; bit4 of FPFLAGS is not-a-number exception flag

   cblock
   FPFLAGS                             ; Exception flags
   SIGN                                ; save location for sign in MSB
   LOOPCOUNT
   endc

;************************************************************
;  Main program storage locations

   cblock
   charsLeftOnLine                     ; Count of characters left on the current line of the LCD
   CHR
   bcd:5                               ; BCD, MSD first

   temperatureTemp:2                   ; Temporary location used by displayTemperature()
   putNibbleTemp                       ; Temporary used by putLCDNibble while assembling bits to send to LCD
   endc

;************************************************************
;  1-wire code scratchpad locations

BITCOUNT equ putNibbleTemp             ; Keep track of bits (overlay on putNibbleTemp as won't use both at once)

   IF putNibbleTemp > h'6F'
   error "### too many bank 0 register definitions!"
   ENDIF ; putNibbleTemp > h'6F'

;************************************************************
   cblock h'70'                        ; Locations common to all register banks (h'70' to h'7F')
   endc

;************************************************************
;  Variables that must be in the register area common to all register banks

;  This section is currently empty

;************************************************************
;  Following variables can go either in this area or in bank 0

   cblock
   interruptCount                      ; Counts where we are in the 20 x 50ms cycle
   contrast                            ; Contrast setting for the LCD. h'FF' = max contrast, h'00' = min contrast
   endc

   cblock
   putNibbleTempPORTB                  ; Temporary used by putLCDNibble while assembling bits to send to LCD

   delay1                              ; used by delay routines
   delay2                              ; used by delay routines
   downCount                           ; Counts how long multifunction button pressed in 50ms units
   endc

   IF downCount > h'7F'
   error "### too many common bank register definitions!"
   ENDIF ; downCount > h'7F'

;************************************************************
;  flags1, flags2, flags3 and flags4 bit definitions
;
;  A flag is a single bit representing a boolean value.
;  True is represented by a '1' bit (bit set)
;  False is represented by a '0' bit (bit clear)

#define displayContrastMenu flags1,0   ; True iff displayContrastSetting() should display the root contrast menu
#define settingUp           flags1,1   ; True iff display contrast is being increased in contrastMenu()
                                       ;   also used in settingMenu() where it is true iff the setting is being increased
#define convertInterrupt    flags1,2   ; True iff the current interrupt is for initiating temperature conversion
#define displayNegative     flags1,3   ; True iff the temperature being displayed is negative (so a minus sign should be written to the display)
#define doDecimalPoint      flags1,4   ; Local to displayTemperature. Used to control display of decimal point
#define oneDecimalPlace     flags1,5   ; Used to control display of decimal point
#define longPress           flags1,6   ; True iff the multifunction button was pressed for at least 1.5 seconds
#define degreeC             flags1,7   ; True iff temperature units are degrees Celsius
                                       ; False iff temperature units are degrees Fahrenheit
#define changedSetting      flags2,0   ; True iff changed settings need to be written back to EEPROM
#define PDBYTE              flags2,1   ; True iff presence detect bit received from DS18B20 temperature sensor
#define addRounding         flags2,3   ; True iff a rounding correction (specified by addPoint5) should be added to the temperature reading
#define addPoint5           flags2,4   ; True if 0.5 rounding correction should be added (with addRounding true), false if 0.05 should be added
#define display             flags4,2   ; False iff processAndDisplay() should not display anything
#define thermostatInvert    flags4,3   ; False iff thermostatOutput should be high iff temperature >= thermostatTemperature
#define thermostatManual    flags4,4   ; True iff thermostatOutput is manually controlled
#define negative            flags4,5   ; Used in roundTemperature
#define hysteresisTemperature flags4,6 ; True iff the 32 degree offset should not be added when converting degrees C to degrees F in displayTemperature()

;************************************************************
;  GENERAL MATH LIBRARY DEFINITIONS

B0      equ   0
B1      equ   1
B2      equ   2
B3      equ   3
B4      equ   4
B5      equ   5
B6      equ   6
B7      equ   7

MSB     equ   7
LSB     equ   0

;************************************************************
;  DS18B20 commands

   constant SKIPROM        = h'CC'     ; Skip ROM command
   constant CONVERT        = h'44'     ; Initiate temperature conversion
   constant READSCRATCHPAD = h'BE'     ; Read the scratchpad memory
   constant READROM        = h'33'     ; Read the ROM (single DS18B20 only)

;************************************************************
;  LCD special characters

   constant degreeSign     = h'DF'     ; Character code for the degree sign in LCD CG ROM

;************************************************************
;  Macros for frequently used code fragments
;************************************************************
;  Select Register Bank 0
bank0: macro
   errorlevel +302                     ; Re-enable bank warning
   banksel PCL                         ; Select Bank 0 (PCL used as a proxy for any bank0 register here)
   endm

;************************************************************
;  bankselNonZero

bankselNonZero: macro label
   errorlevel -302                     ; Disable bank warning
   banksel label
   endm

;************************************************************
;  Select Register Banks 0 and 1 for indirect addressing
;
;  Precondition: nil

bank0indirect: macro
   clrf FSR0H                          ; Select Banks 0 and 1
   endm

;************************************************************
;  Select Register Banks 2 and 3 for indirect addressing
;
;  Precondition: bank 0/1 must be selected for indirect addressing

bank2indirect: macro
   bsf FSR0H,0                         ; Select Banks 2 and 3
   endm

;************************************************************
;  Select Register Banks 4 and 5 for indirect addressing
;
;  Precondition: bank 0/1 must be selected for indirect addressing

bank4indirect: macro
   bsf FSR0H,1                         ; Select Banks 4 and 5
   endm

;************************************************************
;  gotobra dest - becomes 'bra dest' if the processor supports it, otherwise just 'goto dest'
;     Either way, it assembles to a single word instruction, so it is safe to put after a
;     btfsc, btfss, incfsz, or decfsz instruction.
;
;     Good for local jumps, whether following a conditional skip or not, where the limited range of
;     the bra isn't a problem, but the advantage is not having to worry about page select bits.

gotobra: macro dest
   bra dest
   endm

;************************************************************
;  gotops dest - issues page select code for destination dest before goto dest.
;     As a result, dest can be anywhere in program memory.
;
;     Note: this macro is not safe following btfsc, btfss, incfsz and decfsz instructions
;       because these will only skip the page select code, not the goto.

gotops: macro dest
   errorlevel -306
   pagesel dest
   goto dest
   errorlevel +306
   endm

;************************************************************
;  callps dest - issues page select code for destination dest before call dest
;     As a result, dest can be anywhere in program memory.
;
;     Note: this macro is not safe following btfsc, btfss, incfsz and decfsz instructions
;       because these will only skip the page select code, not the call.

callps: macro dest
   errorlevel -306
   pagesel dest
   call dest
   errorlevel +306
   endm

;************************************************************
;  swap8 - swap bytes in register file via W

swap8: macro this,that
   movfw this                          ; get this
   xorwf that,f                        ; swap using Microchip
   xorwf that,W                        ; Tips'n Tricks
   xorwf that,f                        ; #18
   movwf this
   endm

;************************************************************
;  copy8 - copy byte in register file via W

copy8: macro from,to
   movfw from
   movwf to
   endm

;************************************************************
;  copyL8 - assigns the 8 bit literal value 'from' to <to> via W

copyL8: macro from,to
   movlw from
   movwf to
   endm

;************************************************************
;  copy16 - copies the 16 bit value from <Aarg+1:Aarg> into <Barg+1:Barg>

copy16: macro Aarg,Barg
   copy8 Aarg+0,Barg+0
   copy8 Aarg+1,Barg+1
   endm

;************************************************************
;  copyL16 - assigns the 16 bit literal value 'from' to <to+1:to> via W

copyL16: macro from,to
   copyL8 from & h'FF',to+0
   copyL8 from >>d'08' & h'FF',to+1
   endm

;************************************************************
;  copy24 - copies the 24 bit value from <Aarg+2:Aarg+1:Aarg> into <Barg+2:Barg+1:Barg>

copy24: macro Aarg,Barg
   copy8 Aarg+0,Barg+0
   copy8 Aarg+1,Barg+1
   copy8 Aarg+2,Barg+2
   endm

;************************************************************
;  copyL24 - assigns the 24 bit literal value 'from' to <to+2:to+1:to>

copyL24: macro from,to
   copyL8 from & h'FF',to+0
   copyL8 from >>d'08' & h'FF',to+1
   copyL8 from >>d'16' & h'FF',to+2
   endm

;************************************************************
;  copyL32 - assigns the 32 bit literal value from to <to+3:to+2:to+1:to>

copyL32: macro from,to
   copyL8 from & h'FF',to
   copyL8 from >>d'08' & h'FF',to+1
   copyL8 from >>d'16' & h'FF',to+2
   copyL8 from >>d'24' & h'FF',to+3
   endm

;************************************************************
;  clear8 - clears the 1-byte argument

clear8: macro arg
   clrf arg
   endm

;************************************************************
;  clear16 - clears the 2-byte argument

clear16: macro arg
   clear8 arg
   clear8 arg+1
   endm

;************************************************************
;  clear24 - clears the 3-byte argument

clear24: macro arg
   clear8 arg
   clear8 arg+1
   clear8 arg+2
   endm

;************************************************************
;  clear32 - clears the 4-byte argument

clear32: macro arg
   clear8 arg
   clear8 arg+1
   clear8 arg+2
   clear8 arg+3
   endm

;************************************************************
;  invert1 - inverts the single bit in arg, where arg consists of an address,bitnumber pair.

invert1: macro arg1,arg2               ; function invert1(bit) {
   local i1if1else
   local i1if1end
   btfss arg1,arg2                     ;    if (bit) {  // if 1
   gotobra i1if1else
   bcf arg1,arg2                       ;       bit = false;
   gotobra i1if1end
i1if1else:                             ;    } else {
   bsf arg1,arg2                       ;       bit = true;
i1if1end:                              ;    }  // if 1
   endm                                ; }  // invert1()

;************************************************************
;  inc16 - increments the 16 bit value in <Aarg+1:Aarg>
;     Aarg is the LSB, Aarg+1 is the MSB

inc16: macro Aarg
   local inc16end
   incfsz Aarg,f                       ; Increment the LSB
   gotobra inc16end                    ; Skip the following instruction if no overflow
   incf Aarg+1,f                       ; Increment the MSB
inc16end:
   endm

;************************************************************
;  dec16 - decrements the 16 bit value in <Aarg+1:Aarg>
;     Aarg is the LSB, Aarg+1 is the MSB

dec16: macro Aarg
   movlw h'01'                         ; Decrement the LSB
   subwf Aarg,f
   btfss _C
   decf Aarg+1,f                       ; First decrement caused a borrow, so decrement the MSB
   endm

;*******************************************************************************************************
;  add8 reg1,reg2 - adds 8-bit reg2 to reg1 with result in reg1. Uses W.

add8: macro reg1,reg2
   movfw reg2
   addwf reg1,f
   endm

;*******************************************************************************************************
;  add8_L8 reg,literal - adds 8-bit literal to reg with result in reg. Uses W.

add8_L8: macro reg,literal
   movlw literal
   addwf reg,f
   endm

;************************************************************
;  add16_8 - 16 bit plus 8 bit add
;  Barg (8 bits) is added to <Aarg+1:Aarg> (16 bits), with the result in <Aarg+1:Aarg>

add16_8: macro Aarg,Barg
   movfw Barg                          ; Aarg = Aarg + Barg
   addwf Aarg,f
   btfsc _C
   incf Aarg+1,f
   endm

;************************************************************
;  add24_16 - 24 bit plus 16 bit add
;  <Barg+1:Barg> (16 bits) is added to <Aarg+2:Aarg+1:Aarg> (24 bits), with the result in <Aarg+2:Aarg+1:Aarg>

add24_16: macro Aarg,Barg
   movfw Barg                          ; Aarg = Aarg + Barg
   addwf Aarg,f
   movfw Barg+1
   btfsc _C
   incfsz Barg+1,W
   addwf Aarg+1,f
   btfsc _C
   incf Aarg+2,f
   endm

;************************************************************
;  add16_L8 - 16 bit plus 8 bit literal add
;     Literal Barg (8 bits) is added to <Aarg+1:Aarg> (16 bits), with the result in <Aarg+1:Aarg>
;     Uses W

add16_L8: macro Aarg,Barg
   movlw Barg                          ; Aarg = Aarg + Barg
   addwf Aarg,f
   btfsc _C
   incf Aarg+1,f
   endm

;************************************************************
;  add16 - 16 bit add
;     <Barg+1:Barg> (16 bits) is added to <Aarg+1:Aarg> (16 bits), with the result in <Aarg+1:Aarg>

add16: macro Aarg,Barg
   movfw Barg                          ; Aarg = Aarg + Barg
   addwf Aarg,f
   movfw Barg+1
   btfsc _C
   incfsz Barg+1,W
   addwf Aarg+1,f
   endm

;************************************************************
;  add16_L16 - 16 bit plus 16 bit literal add
;     Literal Barg (16 bits) is added to <Aarg+1:Aarg> (16 bits), with the result in <Aarg+1:Aarg>

add16_L16: macro Aarg,Barg
   movlw Barg & h'FF'                  ; Aarg = Aarg + Barg
   addwf Aarg,f
   movlw Barg >>d'08' & h'FF'
   btfsc _C
   incf Aarg+1,f
   addwf Aarg+1,f
   endm

;************************************************************
;  add24 - 24 bit add
;     <Barg+2:Barg+1:Barg> (24 bits) is added to <Aarg+2:Aarg+1:Aarg> (24 bits),
;     with the result in <Aarg+2:Aarg+1:Aarg>

add24: macro Aarg,Barg
   movfw Barg                          ; Aarg = Aarg + Barg
   addwf Aarg,f
   movfw Barg+1
   btfsc _C
   incfsz Barg+1,W
   addwf Aarg+1,f
   movfw Barg+2
   btfsc _C
   incfsz Barg+2,W
   addwf Aarg+2,f
   endm

;************************************************************
;  add32 - 32 bit add
;     <Barg+3:Barg+2:Barg+1:Barg> (32 bits) is added to <Aarg+3:Aarg+2:Aarg+1:Aarg> (32 bits),
;     with the result in <Aarg+3:Aarg+2:Aarg+1:Aarg>

add32: macro Aarg,Barg
   movfw Barg                          ; Aarg = Aarg + Barg
   addwf Aarg,f
   movfw Barg+1
   btfsc _C
   incfsz Barg+1,W
   addwf Aarg+1,f
   movfw Barg+2
   btfsc _C
   incfsz Barg+2,W
   addwf Aarg+2,f
   movfw Barg+3
   btfsc _C
   incfsz Barg+3,W
   addwf Aarg+3,f
   endm

;************************************************************
;  Subtraction table
;
;  The following table is a handy reference when writing code to compare signed and unsigned numbers:
;
;  --------------------------------------------------------------------------------------------------------------------
;  | A binary | B binary | A - B | Carry  | A unsigned | B unsigned | A-B unsigned | A signed | B signed | A-B signed |
;  |----------|--------- |-------| -------| -----------| -----------| -------------| ---------| ---------| -----------|
;  |       00 |       00 |    00 |     1  |          0 |          0 |            0 |        0 |        0 |          0 |
;  |       01 |       00 |    01 |     1  |          1 |          0 |            1 |        1 |        0 |          1 |
;  |       10 |       00 |    10 |     1  |          2 |          0 |            2 |       -2 |        0 |         -2 |
;  |       11 |       00 |    11 |     1  |          3 |          0 |            3 |       -1 |        0 |         -1 |
;  |          |          |       |        |            |            |              |          |          |            |
;  |       00 |       01 |    11 |     0  |          0 |          1 |    -overflow |        0 |        1 |         -1 |
;  |       01 |       01 |    00 |     1  |          1 |          1 |            0 |        1 |        1 |          0 |
;  |       10 |       01 |    01 |     1  |          2 |          1 |            1 |       -2 |        1 |  -overflow |
;  |       11 |       01 |    10 |     1  |          3 |          1 |            2 |       -1 |        1 |         -2 |
;  |          |          |       |        |            |            |              |          |          |            |
;  |       00 |       10 |    10 |     0  |          0 |          2 |    -overflow |        0 |       -2 |  +overflow |
;  |       01 |       10 |    11 |     0  |          1 |          2 |    -overflow |        1 |       -2 |  +overflow |
;  |       10 |       10 |    00 |     1  |          2 |          2 |            0 |       -2 |       -2 |          0 |
;  |       11 |       10 |    01 |     1  |          3 |          2 |            1 |       -1 |       -2 |          1 |
;  |          |          |       |        |            |            |              |          |          |            |
;  |       00 |       11 |    01 |     0  |          0 |          3 |    -overflow |        0 |       -1 |          1 |
;  |       01 |       11 |    10 |     0  |          1 |          3 |    -overflow |        1 |       -1 |          2 |
;  |       10 |       11 |    11 |     0  |          2 |          3 |    -overflow |       -2 |       -1 |         -1 |
;  |       11 |       11 |    00 |     1  |          3 |          3 |            0 |       -1 |       -1 |          0 |
;  --------------------------------------------------------------------------------------------------------------------
;
; For an unsigned subtraction A - B, -overflow, meaning the result is negative, is equivalent to Carry = 0 (+overflow cannot happen).
; In other words, the Carry bit will be 0 if and only if A < B.
;
; For signed subtraction, overflow is equivalent to:
;    (A negative & B positive & A - B positive) | (A positive & B negative & A - B negative)
;
;*******************************************************************************************************
;  sub8 reg1,reg2 - subtracts 8-bit reg2 from reg1 with result in reg1. Uses W.

sub8: macro reg1,reg2
   movfw reg2
   subwf reg1,f
   endm

;*******************************************************************************************************
;  sub8_L8 reg,literal - subtracts 8-bit literal from reg with result in reg. Uses W.

sub8_L8: macro reg,literal
   movlw literal
   subwf reg,f
   endm

;************************************************************
;  sub16_8 - 16 bit minus 8 bit subtract
;     Barg (8 bits) is subtracted from <Aarg+1:Aarg> (16 bits), with the result in <Aarg+1:Aarg>

sub16_8: macro Aarg,Barg
   movfw Barg                          ; Aarg = Aarg - Barg
   subwf Aarg,f
   btfss _C
   decf Aarg+1,f
   endm

;************************************************************
;  sub16_L8 - 16 bit minus 8 bit literal subtract
;     Literal Barg (8 bits) is subtracted from <Aarg+1:Aarg> (16 bits), with the result in <Aarg+1:Aarg>

sub16_L8: macro Aarg,Barg
   movlw Barg                          ; Aarg = Aarg - Barg
   subwf Aarg,f
   btfss _C
   decf Aarg+1,f
   endm

;************************************************************
;  sub16 - 16 bit subtract
;     <Barg+1:Barg> (16 bits) is subtracted from <Aarg+1:Aarg> (16 bits), with the result in <Aarg+1:Aarg>
;     For unsigned arguments, exits with C=0 if result is less than zero (Aarg < Barg),
;                                        C=1 if result is greater than or equal to zero (Aarg >= Barg)

sub16: macro Aarg,Barg
   movfw Barg                          ; Aarg = Aarg - Barg
   subwf Aarg,f
   movfw Barg+1
   btfss _C
   incfsz Barg+1,W
   subwf Aarg+1,f
   endm

;************************************************************
;  sub16_L16 - 16 bit minus 16-bit literal subtract
;     <high Barg:low Barg> (16 bits literal) is subtracted from <Aarg+1:Aarg> (16 bits),
;     with the result in <Aarg+1:Aarg>
;     Exits with C=0 if result is less than zero, C=1 if result is greater than or equal to zero.

sub16_L16: macro Aarg,Barg
   movlw Barg & h'FF'                  ; Move low byte of literal Barg into W
   subwf Aarg,f                        ;   and subtract from low byte of Aarg
   movlw Barg >>d'08' & h'FF'          ; Move high byte of literal Barg into W
   btfss _C                            ; If there was a borrow from the low-byte subtraction,
   movlw (Barg >>d'08' & h'FF')+1      ;   add 1 to W
   subwf Aarg+1,f                      ; Subtract W from high byte of Aarg
   endm

;************************************************************
;  sub24 - 24 bit subtract
;     <Barg+2:Barg+1:Barg> (24 bits) is subtracted from <Aarg+2:Aarg+1:Aarg> (24 bits),
;     with the result in <Aarg+2:Aarg+1:Aarg>
;     For unsigned arguments, exits with C=0 if result is less than zero,
;                                        C=1 if result is greater than or equal to zero.

sub24: macro Aarg,Barg
   movfw Barg                          ; Aarg = Aarg - Barg
   subwf Aarg,f
   movfw Barg+1
   btfss _C
   incfsz Barg+1,W
   subwf Aarg+1,f
   movfw Barg+2
   btfss _C
   incfsz Barg+2,W
   subwf Aarg+2,f
   endm

;************************************************************
;  sub24_16 - 24 bit minus 16 bit subtract
;     <Barg+1:Barg> (16 bits) is subtracted from <Aarg+2:Aarg+1:Aarg> (24 bits),
;     with the result in <Aarg+2:Aarg+1:Aarg>

sub24_16: macro Aarg,Barg
   movfw Barg                          ; Aarg = Aarg - Barg
   subwf Aarg,f
   movfw Barg+1
   btfss _C
   incfsz Barg+1,W
   subwf Aarg+1,f
   btfss _C
   decf Aarg+2,f
   endm

;************************************************************
;  sub24_L24 - 24-bit minus 24-bit literal subtract
;
;     Literal Barg (24 bits) is subtracted from <Aarg+2:Aarg+1:Aarg> (24 bits),
;     with the result in <Aarg+2:Aarg+1:Aarg>
;
;     For unsigned arguments, exits with C=0 if result is less than zero,
;                                        C=1 if result is greater than or equal to zero.

sub24_L24: macro Aarg,Barg
   movlw Barg & h'FF'                  ; Aarg = Aarg - Barg
   subwf Aarg,f
   movlw Barg >>d'08' & h'FF'
   btfss _C
   movlw (Barg >>d'08' & h'FF')+1
   subwf Aarg+1,f
   movfw Barg >>d'16' & h'FF'
   btfss _C
   movlw (Barg >>d'16' & h'FF')+1
   subwf Aarg+2,f
   endm

;************************************************************
;  sub32 - 32 bit subtract
;     <Barg+3:Barg+2:Barg+1:Barg> (32 bits) is subtracted from <Aarg+3:Aarg+2:Aarg+1:Aarg> (32 bits),
;     with the result in <Aarg+3:Aarg+2:Aarg+1:Aarg>
;
;     For unsigned arguments, exits with C=0 if result is less than zero,
;                                        C=1 if result is greater than or equal to zero.

sub32: macro Aarg,Barg
   movfw Barg                          ; Aarg = Aarg - Barg
   subwf Aarg,f
   movfw Barg+1
   btfss _C
   incfsz Barg+1,W
   subwf Aarg+1,f
   movfw Barg+2
   btfss _C
   incfsz Barg+2,W
   subwf Aarg+2,f
   movfw Barg+3
   btfss _C
   incfsz Barg+3,W
   subwf Aarg+3,f
   endm

;************************************************************
;  Compare two 8-bit register file variables via W
;     and skip if equal

compare8skipEqual: macro this,that
   movfw this
   xorwf that,W
   btfss _Z
   endm

;************************************************************
;  Compare two 8-bit register file variables via W
;     and skip if not equal

compare8skipNotEqual: macro this,that
   movfw this
   xorwf that,W
   btfsc _Z
   endm

;************************************************************
;  Compare 8-bit register file variable with 8-bit constant via W
;     and skip if equal

compareL8skipEqual: macro reg,const
   movfw reg
   xorlw const
   btfss _Z
   endm

;************************************************************
;  Compare 8-bit register file variable with 8-bit constant via W
;     and skip if not equal

compareL8skipNotEqual: macro reg,const
   movfw reg
   xorlw const
   btfsc _Z
   endm

;************************************************************
;  compare16gotoIfEqual this,that,label
;
;     Compares two 16-bit variables via W and goes to label if equal, or
;     falls through if not equal.

compare16gotoIfEqual: macro this,that,label
   local fallthrough
   compare8skipEqual this,that
   gotobra fallthrough
   compare8skipNotEqual this+1,that+1
   gotobra label
fallthrough:
   endm

;************************************************************
;  compare16gotoIfNotEqual this,that,label
;
;     Compares two 16-bit variables via W and goes to label if not equal, or
;     falls through if equal.

compare16gotoIfNotEqual: macro this,that,label
   local fallthrough
   compare8skipEqual this,that
   gotobra label
   compare8skipEqual this+1,that+1
   gotobra label
   endm

;************************************************************
;  neg16 - negates its 16-bit argument in place
;
;      <arg+1:arg> <-- -<arg+1:arg>

neg16: macro arg
   local N16if1end
   comf arg+1,f                        ; Complement <arg+1:arg>
   comf arg,f
   incfsz arg,f                        ; and add 1 to arg (LSB)
   gotobra N16if1end
   incf arg+1,f                        ; Deal with carry into arg+1 (MSB)
N16if1end:
   endm

;************************************************************
;  neg32 - negates its 32-bit argument in place
;
;      <arg+3:arg+2:arg+1:arg> <-- -<arg+3:arg+2:arg+1:arg>

neg32: macro arg
   local N32if1end
   comf arg+3,f                        ; Complement <arg+3:arg+2:arg+1:arg>
   comf arg+2,f
   comf arg+1,f
   comf arg,f
   incfsz arg,f                        ; and add 1 to arg (LSB)
   gotobra N32if1end
   incfsz arg+1,f                      ; Deal with carry into arg+1
   gotobra N32if1end
   incfsz arg+2,f
   gotobra N32if1end
   incf arg+3,f
N32if1end:
   endm

;************************************************************
;  abs16 - replaces a 16 bit variable with its absolute value
;
;     <Aarg+1:Aarg> <-- abs(<Aarg+1:Aarg>)

abs16: macro Aarg
   local endAbs16
   btfss Aarg+1,7                      ; Test sign bit
   gotobra endAbs16                    ; Done if clear
   neg16 Aarg
endAbs16:
   endm

;************************************************************
;  read16 - reads a 16-bit value from the file register pointed to by the variable indx
;           and stores in <destination:destination+1>

read16: macro indx,destination
   copy8 indx,FSR0L
   copy8 indx+1,FSR0H
   copy8 INDF0,destination
   incf FSR0L,f
   copy8 INDF0,destination+1
   endm

;************************************************************
;  write16 - writes a 16-bit value from <source:source+1> to the file register pointed to
;            by the 16-bit variable indx

write16: macro indx,source
   copy8 indx,FSR0L
   copy8 indx+1,FSR0H
   copy8 source,INDF0
   incf FSR0L,f
   copy8 source+1,INDF0
   endm

;************************************************************
;  Dallas Semiconductor 1-Wire macros

;************************************************************
;  OW_HIZ - makes the one-wire interface pin high impedance

OW_HIZ: macro
   bankselNonZero TRISA                ; Select Bank 1 of data memory
   bsf DQTRIS                          ; Make DQ pin High Z
   bank0                               ; Select Bank 0 of data memory
   endm

;************************************************************
;  OW_LO - makes the one-wire interface pin low

OW_LO: macro
   bcf DQPRTBIT                        ; Clear the DQ bit. DQPRTBIT is in bank 0
   bankselNonZero TRISA                ; Select Bank 1 of data memory
   bcf DQTRIS                          ; Make DQ pin an output
   bank0                               ; Select Bank 0 of data memory
   endm

;************************************************************
;  usDelay time - delays for time us.
;
;  time must be a multiple of 2us.
;  Minimum delay is 2us
;  Maximum delay is 65536 x 2us = 131ms

usDelay: macro us
   copyL8 high (us / 2 - 1),delay1     ; 2 cycles
   copyL8 low (us / 2 - 1),delay2      ; 2 cycles
   callps delayLoopPlus2
   endm

;************************************************************
;  Reset vector

   org h'0000'
   gotops main

;************************************************************
;  Table of message strings. Keep this in low memory to avoid problems with
;  crossing a 256-program-word boundary

table:
   addwf PCL,f
noSensorString:
   dt "Err"
   dt 0
holdToChangeString:
   dt "Hold to change"
   dt 0
unitString:
   dt "Units: "
   dt degreeSign
   dt 0
contrastString:
   dt "Contrast: "
   dt 0
holdString:
   dt "Hold "
   dt 0
onString:
   dt "on"
   dt 0
offString:
   dt "off"
   dt 0
ONString:
   dt "ON"
   dt 0
OFFString:
   dt "OFF"
   dt 0
switchAtString:
   dt "Switch at"
   dt 0
hysteresisString:
   dt "Hysteresis"
   dt 0
manualString:
   dt "Manual"
   dt 0
AUTOString:
   dt "AUTO"
   dt 0
MANUALString:
   dt "MANUAL"
   dt 0
modeString:
   dt "Mode: "
   dt 0
invertString:
   dt "Invert: "
   dt 0

   constant noSensorStringOffset = noSensorString - table - 1;
   constant holdToChangeStringOffset = holdToChangeString - table - 1;
   constant unitStringOffset = unitString - table - 1;
   constant contrastStringOffset = contrastString - table - 1;
   constant holdStringOffset = holdString - table - 1;
   constant onStringOffset = onString - table - 1;
   constant offStringOffset = offString - table - 1;
   constant ONStringOffset = ONString - table - 1;
   constant OFFStringOffset = OFFString - table - 1;
   constant switchAtStringOffset = switchAtString - table - 1;
   constant hysteresisStringOffset = hysteresisString - table - 1;
   constant manualStringOffset = manualString - table - 1;
   constant AUTOStringOffset = AUTOString - table - 1;
   constant MANUALStringOffset = MANUALString - table - 1;
   constant modeStringOffset = modeString - table - 1;
   constant invertStringOffset = invertString - table - 1;

;************************************************************
;  DisplayMessage - displays a message from the table on the LCD
;
;  Input: W is an offset from the start of the table to the label
;         of the string to display

displayMessage:                        ; function displayMessage() {
   movwf TEMPB0                        ;    TEMPB0 = W;
DMwhile1:                              ;    while (
   movfw TEMPB0                        ;            table[TEMPB0]
   callps table
   andlw h'FF'                         ;            != 0 ) {  // while 1
   btfsc _Z
   return
   callps displayChar                  ;       displayChar(table[TEMPB0]);
   incf TEMPB0,f                       ;       TEMPB0++;
   gotobra DMwhile1                    ;    }  // end while 1
                                       ; }  // displayMessage()

;************************************************************
; Dallas 1-Wire Support for PIC16F1827
;
; Processor has 32MHz clock and 1/8 us per instruction cycle.
;************************************************************

us4:
   copyL8 high 1,delay1                ; 2 cycles
   copyL8 low 1,delay2                 ; 2 cycles
   gotops delayLoopPlus2

;************************************************************
OW_RESET:
   OW_HIZ                              ; Start with the line high
   bcf PDBYTE                          ; Clear the PD flag
   callps us4
   OW_LO
   usDelay d'500'                      ; Drive Low for 500us
   OW_HIZ
   usDelay d'70'                       ; Release line and wait 70us for PD Pulse
   btfss DQPRTBIT                      ; Read for a PD Pulse
   bsf PDBYTE                          ; Set PDBYTE to 1 if get a PD Pulse
   usDelay d'400'                      ; Wait 400us after PD Pulse
   retlw 0

;************************************************************
DSRXBYTE: ; Byte read is stored in TEMP and W
   copyL8 d'8',BITCOUNT                ; Set BITCOUNT equal to 8 to count the bits
DSRXLP:
   OW_LO
   usDelay d'6'
   OW_HIZ
   callps us4
   movfw DQPORT                        ; Read DQ
   andlw 1 << DQBIT                    ; Mask off the DQ bit
   addlw d'255'                        ; C=1 if DQ=1: C=0 if DQ=0
   rrf TEMP,f                          ; Shift C into TEMP
   usDelay d'50'                       ; Wait 50us to end of time slot
   decfsz BITCOUNT,f                   ; Decrement the bit counter
   gotobra DSRXLP
   movfw TEMP                          ; Exit with byte read in W and TEMP
   return

;************************************************************
DSTXBYTE:                              ; Byte to send starts in W
   movwf TEMP                          ; We send it from TEMP
   copyL8 d'8',BITCOUNT                ; Set BITCOUNT equal to 8 to count the bits
DSTXLP:
   OW_LO
   callps us4                          ; Drive the line low for 3us (actually 4us here)
   rrf TEMP,f
   bankselNonZero TRISA                ; Select Bank 1 of data memory
   btfsc STATUS,C                      ; Check the LSB of TEMP for 1 or 0
   bsf DQTRIS                          ; HiZ the line if LSB is 1
   bank0                               ; Select Bank 0 of data memory
   usDelay d'60'                       ; Continue driving line for 60us
   OW_HIZ                              ; Release the line for pullup
   callps us4
   decfsz BITCOUNT,f                   ; Decrement the bit counter
   gotobra DSTXLP
   retlw 0

;************************************************************
;  convertDS - initiates a temperature conversion.
;              The conversion takes 750ms and this subroutine
;              returns before the conversion is complete

convertDS:

; Initialise the DS18B20 (send reset pulse and wait for presence pulse)

   callps OW_RESET

; Send skip ROM command

   movlw SKIPROM
   callps DSTXBYTE

; Send convert command

   movlw CONVERT
   gotops DSTXBYTE

;************************************************************
;  displayCharLiteral char - displays the literal char on the LCD

displayCharLiteral: macro char
   movlw char
   callps displayChar
   endm

;************************************************************
;  Put subroutines below this point
;************************************************************

;************************************************************
displayThermostatOutputState:          ; function displayThermostatOutputState() {
   callps LCDhome                      ;    LCDhome();
   movlw manualStringOffset            ;    displayMessage("Manual: ");
   callps displayMessage
   displayCharLiteral ":"
   displayCharLiteral " "
   gotops displayThermostatState       ;    displayThermostatState();   // ON or OFF according to the state of thermostatOutput. Includes return
                                       ; }  // displayThermostatOutputState()

;************************************************************
thermostatManualMenu:                  ; function thermostatManualMenu() {
   bcf changedSetting                  ;    changedSetting = false;
TMMdoWhile1:                           ;    do {
   callps displayThermostatOutputState ;       displayThermostatOutputState();
   callps holdToChange                 ;       holdToChange();
   btfss longPress                     ;       if (longPress) {  // if 1
;  gotobra TMMif1end
   gotobra TMMdoWhile1end
   invert1 thermostatOutput            ;          thermostatOutput = !thermostatOutput;
   invert1 changedSetting              ;          changedSetting = !changedSetting;
   gotobra TMMdoWhile1
;TMMif1end:                            ;       }  // if 1
                                       ;    } while (longPress);
TMMdoWhile1end:
   btfss changedSetting                ;    if (changedSetting) {  // if 2
   return
;  gotops writeThermostatOutputToEEPROM;       writeThermostatOutputToEEPROM();  // Fallthrough. Includes return
                                       ;    }  // if 2
                                       ; }  // thermostatManualMenu()

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
writeThermostatOutputToEEPROM:         ; function writeThermostatOutputToEEPROM() {
                                       ;    if (thermostatOutput) {  // if 1
                                       ;       EEPROM[thermostatStateEEPROMaddr] == "1";
                                       ;    } else {  // if 1
                                       ;       EEPROM[thermostatStateEEPROMaddr] == "0";
                                       ;    }  // if 1
   movlw thermostatStateEEPROMaddr-EEPROMstart
   bankselNonZero EEADR
   movwf EEADR
   bank0
   movlw "1"
   btfss thermostatOutput
   movlw "0"
   gotops writeEEPROM                  ;       // Includes return
                                       ; }  // writeThermostatOutputToEEPROM()

;************************************************************
thermostatAutoManualMenu:              ; function thermostatAutoManualMenu() {
   bcf changedSetting                  ;    changedSetting = false;
TAMMdoWhile1:                          ;    do {
   callps LCDhome                      ;       LCDhome();
   movlw modeStringOffset              ;       displayMessage("Mode: ");
   callps displayMessage
   callps displayThermostatMode        ;       displayThermostatMode();
   callps holdToChange                 ;       holdToChange();
   btfss longPress                     ;       if (longPress) {  // if 1
;  gotobra TAMMif2end
   gotobra TAMMdoWhile1end
   invert1 thermostatManual            ;          thermostatManual = !thermostatManual;
   invert1 changedSetting              ;          changedSetting = !changedSetting;
   gotobra TAMMdoWhile1
;TAMMif2end:                           ;       }  // if 1
                                       ;    } while (longPress);
TAMMdoWhile1end:
   btfss changedSetting                ;    if (changedSetting) {  // if 2
   return
                                       ;       writeThermostatManualToEEPROM();   // Fallthrough. Includes return
                                       ;    }  // if 2
                                       ; }  // thermostatAutoManualMenu()

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
writeThermostatManualToEEPROM:         ; function writeThermostatManualToEEPROM() {
                                       ;    if (thermostatManual) {  // if 1
                                       ;       EEPROM[thermostatManualEEPROMaddr] == "M";
                                       ;    } else {  // if 1
                                       ;       EEPROM[thermostatManualEEPROMaddr] == "A";
                                       ;    }  // if 1
   movlw thermostatManualEEPROMaddr-EEPROMstart
   bankselNonZero EEADR
   movwf EEADR
   bank0
   movlw "M"
   btfss thermostatManual
   movlw "A"
   gotops writeEEPROM                  ;    // Includes return
                                       ; }  // writeThermostatManualToEEPROM()

;************************************************************
thermostatInvertMenu:                  ; function thermostatInvertMenu() {
   bcf changedSetting                  ;    changedSetting = false;
TIMdoWhile1:                           ;    do {
   callps LCDhome                      ;       LCDhome();
   movlw invertStringOffset            ;       displayMessage("Invert: ");
   callps displayMessage
   callps displayThermostatInvert      ;       displayThermostatInvert();   // ON or OFF according to the state of thermostatInvert
   callps holdToChange                 ;       holdToChange();
   btfss longPress                     ;       if (longPress) {  // if 1
;  gotobra TIMif2end
   gotobra TIMdoWhile1end
   invert1 thermostatInvert            ;          thermostatInvert = !thermostatInvert;
   invert1 changedSetting              ;          changedSetting = !changedSetting;
   gotobra TIMdoWhile1
;TIMif2end:                            ;       }  // if 1
                                       ;    } while (longPress);
TIMdoWhile1end:
   btfss changedSetting                ;    if (changedSetting) {  // if 2
   return
                                       ;       if (thermostatInvert) {  // if 3
                                       ;          EEPROM[thermostatInvertEEPROMaddr] == "I";
                                       ;       } else {  // if 3
                                       ;          EEPROM[thermostatInvertEEPROMaddr] == " ";
                                       ;       }  // if 3
   movlw thermostatInvertEEPROMaddr-EEPROMstart
   bankselNonZero EEADR
   movwf EEADR
   bank0
   movlw "I"
   btfss thermostatInvert
   movlw " "
   gotops writeEEPROM
                                       ;    }  // if 2
                                       ; }  // thermostatInvertMenu()

;************************************************************
holdToChange:                          ; function holdToChange() {
   callps line2                        ;    line2();
   movlw holdToChangeStringOffset      ;    displayMessage("Hold to change");
   callps displayMessage
   callps clearToEndOfLine             ;    clearToEndOfLine();
   gotops waitForPress                 ;    waitForPress();  // includes return
                                       ; }  // holdToChange()

;************************************************************
thermostatMenu:                        ; function thermostatMenu() {
   callps thermostatAutoManualMenu     ;    thermostatAutoManualMenu();
   btfss thermostatManual              ;    if (thermostatManual) {  // if 1
   gotobra thMif1else
   gotops thermostatManualMenu         ;       thermostatManualMenu();      // Includes return
thMif1else:                            ;    }  else { // if 1
   callps thermostatTemperatureMenu    ;       thermostatTemperatureMenu();
   callps thermostatHysteresisMenu     ;       thermostatHysteresisMenu();
   gotops thermostatInvertMenu         ;       thermostatInvertMenu();      // Includes return
;thMif1end:                            ;    }  // if 1
                                       ; }  // thermostatMenu()

;************************************************************
thermostatTemperatureMenu:             ; function thermostatTemperatureMenu() {
                                       ;    setting = thermostatTemperature;
   copy16 thermostatTemperature,setting
   copyL16 -d'880',minSetting          ;    minSetting = -55 * 16;
   copyL16 d'2000',maxSetting          ;    maxSetting = 125 * 16;

   copyL8 switchAtStringOffset,settingNameOffset
   bcf hysteresisTemperature           ;    hysteresisTemperature = false;
   callps settingMenu                  ;    settingMenu(setting=thermostatTemperature,minSetting=-55*16,maxSetting=125*16,settingNameOffset=switchAtStringOffset);
   btfss changedSetting                ;    if (changedSetting) {  // if 1
;  gotobra TTMif1end
   return
   copy16 setting,AARGB1               ;       thermostatTemperature = roundTemperature(thermostatTemperature);
   callps roundTemperature
   copy16 AARGB1,thermostatTemperature
                                       ;       EEPROM[thermostatTemperatureHEEPROMaddr] = thermostatTemperatureH;
   movlw thermostatTemperatureHEEPROMaddr-EEPROMstart
   bankselNonZero EEADR
   movwf EEADR
   bank0
   movfw thermostatTemperatureH
   callps writeEEPROM
                                       ;       EEPROM[thermostatTemperatureLEEPROMaddr] = thermostatTemperatureL;
   movlw thermostatTemperatureLEEPROMaddr-EEPROMstart
   bankselNonZero EEADR
   movwf EEADR
   bank0
   movfw thermostatTemperatureL
   gotops writeEEPROM                  ;        // Includes return
TTMif1end:                             ;    }  // if 1
;  return                              ; }  // thermostatTemperatureMenu()

;************************************************************
thermostatHysteresisMenu:              ; function thermostatHysteresisMenu() {
                                       ;    setting = thermostatHysteresis;
   copy16 thermostatHysteresis,setting
   copyL16 d'0',minSetting             ;    minSetting = 0;
   copyL16 d'2880',maxSetting          ;    maxSetting = 180;
                                       ;
   bsf hysteresisTemperature           ;    hysteresisTemperature = true;
   copyL8 hysteresisStringOffset,settingNameOffset
   callps settingMenu                  ;    settingMenu(setting=thermostatHysteresis,minSetting=0,maxSetting=180*16,settingNameOffset=hysteresisStringOffset);
   btfss changedSetting                ;    if (changedSetting) {  // if 1
   return
;  gotobra THMif1end
   copy16 setting,AARGB1               ;       thermostatHysteresis = roundTemperature(thermostatHysteresis);
   callps roundTemperature
   copy16 AARGB1,thermostatHysteresis
                                       ;       EEPROM[thermostatHysteresisHEEPROMaddr] = thermostatHysteresisH;
   movlw thermostatHysteresisHEEPROMaddr-EEPROMstart
   bankselNonZero EEADR
   movwf EEADR
   bank0
   movfw thermostatHysteresisH
   callps writeEEPROM
                                       ;       EEPROM[thermostatHysteresisLEEPROMaddr] = thermostatHysteresisL;
   movlw thermostatHysteresisLEEPROMaddr-EEPROMstart
   bankselNonZero EEADR
   movwf EEADR
   bank0
   movfw thermostatHysteresisL
   gotops writeEEPROM                  ;       // Includes return
;THMif1end:                            ;    }  // if 1
;  return                              ; }  // thermostatHysteresisMenu()

   constant maxStage = d'8'            ; Upper limit of stage.
   constant settingChanges = d'48'     ; How many 50ms setting change periods before stage is incremented

;************************************************************
;  contrastMenu: displays a menu screen with the contrast setting and either HOLD + or HOLD - on the second line.
;                Pressing and holding the multifunction button increments or decrements the setting, as appropriate.
;                If the setting is changed it is written back to EEPROM
;
;                Inputs:
;                  contrast           - 8 bit unsigned value
;
;                Local variables:
;                  settingDelta       - amount by which the setting is due to change on the next increment or decrement
;                  settingUp          - true iff the setting value is being increased
;                  changesSkipped     - 8 bits. count of number of short presses of the multifunction button. Adjustment of the setting stops when this reaches 2
;                  stage              - 8 bits. an index that records how fast the setting is being changed
;                  changeCount        - 8 bits. count how many changes to the value of the setting since stage was last incremented
;
;                Outputs:
;                  contrast           - the input variable is modified by this function
;                  changedSetting     - true iff the contrast setting has been changed (requiring it to be written back to EEPROM)

contrastMenu:                          ; function contrastMenu() {
   bcf changedSetting                  ;    changedSetting = false;      // True iff setting has changed and therefore has to be written back to EEPROM
contrastMenuChangedSetting:
   callps LCDhome                      ;    LCDhome();
   callps applyContrastSetting         ;    applyContrastSetting();
   movlw contrastStringOffset          ;    displayMessage("Contrast: ");
   callps displayMessage
   copy8 contrast,AARGB3               ;    displayInteger(contrast);
   callps displayInteger8
   callps holdToChange                 ;    holdToChange();

   btfss longPress                     ;    if (longPress) {  // if 1
   gotobra CMif1end
   bsf settingUp                       ;       settingUp = true;
CMdoWhile1:                            ;       do {  // doWhile 1
   callps LCDhome                      ;          LCDhome();
   movlw contrastStringOffset          ;          displayMessage("Contrast: ");
   callps displayMessage
   copy8 contrast,AARGB3               ;          displayInteger(contrast);
   callps displayInteger8
   callps line2                        ;          line2();
   movlw holdStringOffset              ;          displayMessage("Hold ");
   callps displayMessage
   movlw "+"                           ;          if (settingUp) {  // if 1b
   btfss settingUp                     ;             displayChar("+");
   movlw "-"                           ;          } else {
   callps displayChar                  ;             displayChar("-");
                                       ;          }  // if 1b

   callps clearToEndOfLine             ;          clearToEndOfLine();
   callps waitForPress                 ;          waitForPress();
   btfss longPress                     ;          if (longPress) {  // if 2
   gotobra CMif2else
   clrf changesSkipped                 ;             changesSkipped = 0;
   copyL16 d'1',settingDelta           ;             settingDelta = 1;     // How much to increment or decrement the setting by - doubles for each stage
   copyL8 d'1',stage                   ;             stage = 1;
   copyL8 settingChanges,changeCount   ;             changeCount = settingChanges;
CMwhile1:
   btfsc multifunctionButton           ;             while (buttonPressed) {  // while 1
   gotobra CMwhile1end
   bsf changedSetting                  ;                changedSetting = true;     // True iff setting has changed and therefore has to be written back to EEPROM
   callps bgTimeSlice                  ;                bgTimeSlice();   // 50ms delay between checking the button state
   bsf display                         ;                display = true;

                                       ;                // contrast ranges from 0 to 255 and is stored as 8 bits unsigned

   btfss settingUp                     ;                if (settingUp) {  // if 4
   gotobra CMif4else
   copy8 contrast,AARGB0               ;                   if (contrast + settingDelta <= d'255') {  // if 5
   add8 AARGB0,settingDelta
   btfsc _C
   gotobra CMif5else
   add8 contrast,settingDelta          ;                      contrast += settingDelta;
   gotobra CMif5end
CMif5else:                             ;                   } else {  if 5
   copyL8 d'255',contrast              ;                      contrast = d'255';
CMif5end:                              ;                   }  // if 5
   gotobra CMif4end
CMif4else:                             ;                }  else {  // if 4
   copy8 contrast,AARGB0               ;                   if (contrast >= settingDelta) {  // if 6
   sub8 AARGB0,settingDelta
   btfss _C
   gotobra CMif6else
   sub8 contrast,settingDelta          ;                      contrast -= settingDelta;
   gotobra CMif6end
CMif6else:                             ;                   } else {  if 6
   clrf contrast                       ;                      contrast = 0;
CMif6end:                              ;                   }  // if 6
CMif4end:                              ;                }  // end if 4 ELSE
   compareL8skipNotEqual stage,maxStage;                if (stage != maxStage) {  // if 7
   gotobra CMif7end
   decfsz changeCount,f                ;                   changeCount--;
   gotobra CMif8end                    ;                   if (changeCount == 0) {  // if 8
   copyL8 settingChanges,changeCount   ;                      changeCount = settingChanges;
   incf stage,f                        ;                      stage++;
   lslf settingDelta,f                 ;                      settingDelta = settingDelta * 2;
CMif8end:                              ;                   }  // if 8
CMif7end:                              ;                }  // if 7
   callps LCDhome                      ;                LCDhome();
   callps applyContrastSetting         ;                applyContrastSetting();
   movlw contrastStringOffset          ;                displayMessage("Contrast: ");
   callps displayMessage
   copy8 contrast,AARGB3               ;                displayInteger(contrast);
   callps displayInteger8
   callps line2                        ;                line2();
   movlw holdStringOffset              ;                displayMessage("Hold ");
   callps displayMessage
                                       ;                // stageCounter is in the high 4 bits of stage, to save using another register for stageCounter
   movfw stage                         ;                for (stageCounter = stage; stageCounter > 0; stageCounter--) {  // for 1
   lslf stage,f
   lslf stage,f
   lslf stage,f
   lslf stage,f
   iorwf stage,f
CMfor1:
                                       ;                   if (settingUp) {  // if 9
                                       ;                      displayChar("+");
                                       ;                   } else {  // if 9
                                       ;                      displayChar("-");
                                       ;                   }  // if 9
   movlw "+"
   btfss settingUp
   movlw "-"
   callps displayChar
   sub8_L8 stage,h'10'                 ;                }  // for 1
   movfw stage
   andlw h'F0'
   btfss _Z
   gotobra CMfor1
CMfor1end:
   callps clearToEndOfLine             ;                clearToEndOfLine();
   gotops CMwhile1                     ;             }  // end while 1
CMif2else:                             ;          } else {  // if 2
   invert1 settingUp                   ;             settingUp = !settingUp;
   incf changesSkipped,f               ;             changesSkipped++;
CMif2end:                              ;          }  // end if 2
CMwhile1end:
                                       ;       } while (changesSkipped != 2);  // end doWhile 1
   compareL8skipEqual changesSkipped,d'2'
   gotobra CMdoWhile1
CMif1end:                              ;    }  // end if 1
   btfss changedSetting                ;    if (changedSetting) {  // if 10
   return
                                       ;       EEPROM[contrastEEPROMaddr] = contrast;
   movlw contrastEEPROMaddr-EEPROMstart
   bankselNonZero EEADR
   movwf EEADR
   bank0
   movfw contrast
   gotops writeEEPROM
;  return                              ;    }  // end if 10
;                                      ; }  // contrastMenu()

;************************************************************
;  settingMenu: displays a menu screen with the name and value of a temperature setting and a temperature unit on the first line,
;               and either HOLD + or HOLD - on the second line.
;
;               Pressing and holding the multifunction button increments or decrements the setting, as appropriate.
;
;               Inputs:
;                 setting               - 16 bit 2s complement setting value
;                 minSetting            - 16 bits. Minimum value of the setting
;                 maxSetting            - 16 bits. Maximum value of the setting
;                 settingNameOffset     - 8 bits. Offset into the string table of the text preceding the setting value,
;                                         i.e. either switchAtStringOffset for the text "Switch at" or hysteresisStringOffset for the text "Hysteresis".
;                 hysteresisTemperature - Flag. True iff degree F temperatures should be calculated without adding the 32 degree offset
;               Local variables:
;                 settingDelta          - 16 bits. Amount by which the setting is due to change on the next increment or decrement
;                 settingUp             - true iff the setting value is being increased
;                 changesSkipped        - 8 bits. Count of number of short presses of the multifunction button. Adjustment of the setting stops when this reaches 2
;                 stage                 - 8 bits. An index that records how fast the setting is being changed
;                 changeCount           - 8 bits. Count how many changes to the value of the setting since stage was last incremented
;
;               Outputs:
;                 setting               - the input variable is modified by this function
;                 changedSetting        - true iff the setting value has been changed (requiring it to be written back to EEPROM)

settingMenu:                           ; function settingMenu() {
   bcf changedSetting                  ;    changedSetting = false;      // True iff setting has changed and therefore has to be written back to EEPROM
   callps LCDhome                      ;    LCDhome();
   movfw settingNameOffset             ;    displayMessage(settingNameOffset);
   callps displayMessage
   displayCharLiteral " "              ;    displayChar(" ");
   bcf oneDecimalPlace                 ;    displayTemperature(setting);
   copy16 setting,AARGB1
   callps displayTemperature
   callps displayTemperatureUnit       ;    displayTemperatureUnit();
   callps clearToEndOfLine             ;    clearToEndOfLine();
   callps holdToChange                 ;    holdToChange();

   btfss longPress                     ;    if (longPress) {  // if 1
   gotobra SMif1end
   bsf settingUp                       ;       settingUp = true;
   clrf changesSkipped                 ;       changesSkipped = 0;
SMdoWhile1:                            ;       do {  // doWhile 1
   callps LCDhome                      ;          LCDhome();
   movfw settingNameOffset             ;          displayMessage(settingNameOffset);
   callps displayMessage
   displayCharLiteral " "              ;          displayChar(" ");
   bcf oneDecimalPlace                 ;          displayTemperature(setting);
   copy16 setting,AARGB1
   callps displayTemperature
   callps clearToEndOfLine             ;          clearToEndOfLine();
   callps line2                        ;          line2();
   movlw holdStringOffset              ;          displayMessage("Hold ");
   callps displayMessage
   movlw "+"                           ;          if (settingUp) {  // if 1b
   btfss settingUp                     ;             displayChar("+");
   movlw "-"                           ;          } else {
   callps displayChar                  ;             displayChar("-");
                                       ;          }  // if 1b

   callps clearToEndOfLine             ;          clearToEndOfLine();
   callps waitForPress                 ;          waitForPress();
   btfss longPress                     ;          if (longPress) {  // if 2
   gotobra SMif2else
   clrf changesSkipped                 ;             changesSkipped = 0;
   copyL16 d'1',settingDelta           ;             settingDelta = 1;     // How much to increment or decrement the setting by - doubles for each stage
   copyL8 d'1',stage                   ;             stage = 1;
   copyL8 settingChanges,changeCount   ;             changeCount = settingChanges;
SMwhile1:
   btfsc multifunctionButton           ;             while (buttonPressed) {  // while 1
   gotobra SMwhile1end
   bsf changedSetting                  ;                changedSetting = true;     // True iff setting has changed and therefore has to be written back to EEPROM
   callps bgTimeSlice                  ;                bgTimeSlice();   // 50ms delay between checking the button state
   bsf display                         ;                display = true;

   btfss settingUp                     ;                if (settingUp) {  // if 4
   gotobra SMif4else
   copy16 maxSetting,AARGB1            ;                   if (setting + settingDelta <= maxSetting) {  // if 5
   sub16 AARGB1,settingDelta
   sub16 AARGB1,setting
   btfsc AARGB0,7
   gotobra SMif5else
   add16 setting,settingDelta          ;                      setting += settingDelta;
   gotobra SMif5end
SMif5else:                             ;                   } else {  if 5
   copy16 maxSetting,setting           ;                      setting = maxSetting;
SMif5end:                              ;                   }  // if 5
   gotobra SMif4end
SMif4else:                             ;                }  else {  // if 4
   copy16 setting,AARGB1               ;                   if (setting - settingDelta >= minSetting) {  // if 6
   sub16 AARGB1,settingDelta
   sub16 AARGB1,minSetting
   btfsc AARGB0,7
   gotobra SMif6else
   sub16 setting,settingDelta          ;                      setting -= settingDelta;
   gotobra SMif6end
SMif6else:                             ;                   } else {  if 6
   copy16 minSetting,setting           ;                      setting = minSetting;
SMif6end:                              ;                   }  // if 6
SMif4end:                              ;                }  // end if 4 ELSE
                                       ;
   compareL8skipNotEqual stage,maxStage;                if (stage != maxStage) {  // if 7
   gotobra SMif7end
   decfsz changeCount,f                ;                   changeCount--;
   gotobra SMif8end                    ;                   if (changeCount == 0) {  // if 8
   copyL8 settingChanges,changeCount   ;                      changeCount = settingChanges;
   incf stage,f                        ;                      stage++;
   lslf settingDelta,f                 ;                      settingDelta = settingDelta * 2;
   rlf settingDelta+1,f
SMif8end:                              ;                   }  // if 8
SMif7end:                              ;                }  // if 7
   callps LCDhome                      ;                LCDhome();
   movfw settingNameOffset             ;                displayMessage(settingNameOffset);
   callps displayMessage
   displayCharLiteral " "              ;                displayChar(" ");
   bcf oneDecimalPlace                 ;                displayTemperature(setting);
   copy16 setting,AARGB1
   callps displayTemperature
   callps clearToEndOfLine             ;                clearToEndOfLine();
   callps line2                        ;                line2();
   movlw holdStringOffset              ;                displayMessage("Hold ");
   callps displayMessage
                                       ;                // stageCounter is in the high 4 bits of stage, to save using another register for stageCounter
   movfw stage                         ;                for (stageCounter = stage; stageCounter > 0; stageCounter--) {  // for 1
   lslf stage,f
   lslf stage,f
   lslf stage,f
   lslf stage,f
   iorwf stage,f
SMfor1:
                                       ;                   if (settingUp) {  // if 9
                                       ;                      displayChar("+");
                                       ;                   } else {  // if 9
                                       ;                      displayChar("-");
                                       ;                   }  // if 9
   movlw "+"
   btfss settingUp
   movlw "-"
   callps displayChar
   sub8_L8 stage,h'10'                 ;                }  // for 1
   movfw stage
   andlw h'F0'
   btfss _Z
   gotobra SMfor1
SMfor1end:
   callps clearToEndOfLine             ;                clearToEndOfLine();
   gotops SMwhile1                     ;             }  // end while 1
SMif2else:                             ;          } else {  // if 2
   invert1 settingUp                   ;             settingUp = !settingUp;
   incf changesSkipped,f               ;             changesSkipped++;
SMif2end:                              ;          }  // end if 2
SMwhile1end:
                                       ;       } while (changesSkipped != 2);  // end doWhile 1
   compareL8skipEqual changesSkipped,d'2'
   gotobra SMdoWhile1
SMif1end:                              ;    }  // end if 1
   return
;                                      ; }  // settingMenu()

;************************************************************
;  readSettingsFromEEPROM - reads the settings from the EEPROM into file memory
;    To make the program robust in the face of possible corruption of the data in the EEPROM,
;    if a value isn't recognised just set it to a sensible default.
;    This will correct it for next time, or if the EEPROM is permanently faulty
;    at least the user can manually change the setting(s).

readSettingsFromEEPROM:                ; function readSettingsFromEEPROM() {
                                       ;    contrast = EEPROM[contrastEEPROMaddr];
   movlw contrastEEPROMaddr-EEPROMstart
   callps readEEPROM
   movwf contrast

   movlw unitEEPROMaddr-EEPROMstart    ;    degreeC = (EEPROM[unitEEPROMaddr] == "C");

   callps readEEPROM
   bcf degreeC
   xorlw "C"
   btfsc _Z
   bsf degreeC
                                       ;    thermostatTemperature = EEPROM[thermostatTemperatureEEPROMaddr];
   movlw thermostatTemperatureHEEPROMaddr-EEPROMstart
   callps readEEPROM
   movwf thermostatTemperatureH
   movlw thermostatTemperatureLEEPROMaddr-EEPROMstart
   callps readEEPROM
   movwf thermostatTemperatureL
                                       ;    thermostatHysteresis = EEPROM[thermostatHysteresisEEPROMaddr];
   movlw thermostatHysteresisHEEPROMaddr-EEPROMstart
   callps readEEPROM
   movwf thermostatHysteresisH
   movlw thermostatHysteresisLEEPROMaddr-EEPROMstart
   callps readEEPROM
   movwf thermostatHysteresisL
                                       ;    thermostatInvert = (EEPROM[thermostatInvertEEPROMaddr] == "I");
   movlw thermostatInvertEEPROMaddr-EEPROMstart
   callps readEEPROM
   bcf thermostatInvert
   xorlw "I"                           ;       // Valid values are: "I" for invert, " " for no inversion
   btfsc _Z
   bsf thermostatInvert
                                       ;    thermostatManual = (EEPROM[thermostatManualEEPROMaddr] == "M");
   movlw thermostatManualEEPROMaddr-EEPROMstart
   callps readEEPROM
   bcf thermostatManual
   xorlw "M"                           ;       // Valid values are: "M" for manual, "A" for auto
   btfsc _Z
   bsf thermostatManual
   btfss thermostatManual              ;    if (thermostatManual) {  // if 2
   gotobra RSFEif2end
                                       ;       thermostatState = (EEPROM[thermostatStateEEPROMaddr] == "1");
   movlw thermostatStateEEPROMaddr-EEPROMstart
   callps readEEPROM
   xorlw "0"                           ;       // Valid values are: "0" for low, "1" for high
   btfsc _Z
   gotobra RSFEif3else
   bsf thermostatOutput
   gotobra RSFEif3end
RSFEif3else:
   bcf thermostatOutput
RSFEif3end:
                                       ;    }  // if 2
RSFEif2end:
   return
                                       ; }  // readSettingsFromEEPROM()

;************************************************************
;  shiftLeft24 - shifts the 24-bit value <AARGB1:AARGB2:AARGB3> left one place, with a zero going into bit 0.

shiftLeft24:
   lslf AARGB3,f
   rlf AARGB2,f
   rlf AARGB1,f
   return

;************************************************************
;  waste2cycles: just uses 2 cycles

waste2cycles: macro
   local next
   gotobra next
next:
   endm

;************************************************************
;       16x16 Bit Multiplication Macro

SMUL1616L: macro

;    Max Timing:     2+11+6*16+15+2+6*17+16+5 = 249 clks
;    Min Timing:     2+7*6+5+2+6*6+5+4 = 96 clks
;    PM: 55            DM: 9

   copyL8 d'8',LOOPCOUNT

LOOPSM1616A:
   rrf BARGB1,f
   btfsc _C
   gotobra ALSM1616NA
   decfsz LOOPCOUNT,f
   gotobra LOOPSM1616A

   copyL8 d'7',LOOPCOUNT

LOOPSM1616B
   rrf BARGB0,f
   btfsc _C
   gotobra BLSM1616NA
   decfsz LOOPCOUNT,f
   gotobra LOOPSM1616B

   clrf AARGB0
   clrf AARGB1
   retlw h'00'

ALOOPSM1616
   rrf BARGB1,f
   btfss _C
   gotobra ALSM1616NA
   movfw TEMPB1
   addwf AARGB1,f
   movfw TEMPB0
   btfsc _C
   incfsz TEMPB0,W
   addwf AARGB0,f

ALSM1616NA:
   rlf SIGN,W
   rrf AARGB0,f
   rrf AARGB1,f
   rrf AARGB2,f
   decfsz LOOPCOUNT,f
   gotobra ALOOPSM1616

   copyL8 d'7',LOOPCOUNT

BLOOPSM1616:
   rrf BARGB0,f
   btfss _C
   gotobra BLSM1616NA
   movfw TEMPB1
   addwf AARGB1,f
   movfw TEMPB0
   btfsc _C
   incfsz TEMPB0,W
   addwf AARGB0,f

BLSM1616NA:
   rlf SIGN,W
   rrf AARGB0,f
   rrf AARGB1,f
   rrf AARGB2,f
   rrf AARGB3,f
   decfsz LOOPCOUNT,f
   gotobra BLOOPSM1616

   rlf SIGN,W
   rrf AARGB0,f
   rrf AARGB1,f
   rrf AARGB2,f
   rrf AARGB3,f
   endm

;************************************************************
;       16x16 Bit Signed Fixed Point Multiply 16x16 -> 32
;
;       Input:  16 bit signed fixed point multiplicand in <AARGB0:AARGB1>
;               16 bit signed fixed point multiplier in <BARGB0:BARGB1>
;
;       Use:    callps  FXM1616S
;
;       Output: 32 bit signed fixed point product in <AARGB0:AARGB1:AARGB2:AARGB3>
;
;       Result: AARG  <--  AARG x BARG
;
;       Max Timing:     9+249+2 = 260 clks                B > 0
;                       18+249+2 = 269 clks               B < 0
;
;       Min Timing:     9+96 = 105 clks
;
;       PM: 18+55+1 = 74              DM: 9

FXM1616S:
   clrf AARGB2                        ; clear partial product
   clrf AARGB3
   clrf SIGN
   movfw AARGB0
   iorwf AARGB1,W
   btfsc _Z
   retlw h'00'

   movfw AARGB0
   xorwf BARGB0,W
   movwf TEMPB0
   btfsc TEMPB0,MSB
   comf SIGN,f

   btfss BARGB0,MSB
   gotobra M1616SOK

   comf BARGB1,f
   comf BARGB0,f
   incf BARGB1,f
   btfsc _Z
   incf BARGB0,f

   comf AARGB1,f
   comf AARGB0,f
   incf AARGB1,f
   btfsc _Z
   incf AARGB0,f

   btfsc BARGB0,MSB
   gotobra M1616SX

M1616SOK:
   movfw AARGB0
   movwf TEMPB0
   movfw AARGB1
   movwf TEMPB1

   SMUL1616L

   retlw h'00'

M1616SX:
   clrf AARGB2
   clrf AARGB3
   rlf SIGN,W
   rrf AARGB0,f
   rrf AARGB1,f
   rrf AARGB2,f

   retlw h'00'

;************************************************************
;       32/16 Bit Division Macro

SDIV3216L: macro

;  Max Timing:     9+6*17+16+16+6*17+16+16+6*17+16+16+6*17+16+8 = 537 clks
;  Min Timing:     9+6*16+15+15+6*16+15+15+6*16+15+15+6*16+15+3 = 501 clks
;  PM: 157                                 DM: 9

   movfw BARGB1
   subwf REMB1,f
   movfw BARGB0
   btfss _C
   incfsz BARGB0,W
   subwf REMB0,f
   rlf AARGB0,f

   movlw d'7'
   movwf LOOPCOUNT

LOOPS3216A:
   rlf AARGB0,W
   rlf REMB1,f
   rlf REMB0,f
   movfw BARGB1
   btfss AARGB0,LSB
   gotobra SADD26LA

   subwf REMB1,f
   movfw BARGB0
   btfss _C
   incfsz BARGB0,W
   subwf REMB0,f
   gotobra SOK26LA

SADD26LA:
   addwf REMB1,f
   movfw BARGB0
   btfsc _C
   incfsz BARGB0,W
   addwf REMB0,f

SOK26LA:
   rlf AARGB0,f

   decfsz LOOPCOUNT,f
   gotobra LOOPS3216A

   rlf AARGB1,W
   rlf REMB1,f
   rlf REMB0,f
   movfw BARGB1
   btfss AARGB0,LSB
   gotobra SADD26L8

   subwf REMB1,f
   movfw BARGB0
   btfss _C
   incfsz BARGB0,W
   subwf REMB0,f
   gotobra SOK26L8

SADD26L8:
   addwf REMB1,f
   movfw BARGB0
   btfsc _C
   incfsz BARGB0,W
   addwf REMB0,f

SOK26L8:
   rlf AARGB1,f
   movlw d'7'
   movwf LOOPCOUNT

LOOPS3216B:
   rlf AARGB1,W
   rlf REMB1,f
   rlf REMB0,f
   movfw BARGB1
   btfss AARGB1,LSB
   gotobra SADD26LB

   subwf REMB1,f
   movfw BARGB0
   btfss _C
   incfsz BARGB0,W
   subwf REMB0,f
   gotobra SOK26LB

SADD26LB:
   addwf REMB1,f
   movfw BARGB0
   btfsc _C
   incfsz BARGB0,W
   addwf REMB0,f

SOK26LB:
   rlf AARGB1,f

   decfsz LOOPCOUNT,f
   gotobra LOOPS3216B

   rlf AARGB2,W
   rlf REMB1,f
   rlf REMB0,f
   movfw BARGB1
   btfss AARGB1,LSB
   gotobra SADD26L16

   subwf REMB1,f
   movfw BARGB0
   btfss _C
   incfsz BARGB0,W
   subwf REMB0,f
   gotobra SOK26L16

SADD26L16:
   addwf REMB1,f
   movfw BARGB0
   btfsc _C
   incfsz BARGB0,W
   addwf REMB0,f

SOK26L16:
   rlf AARGB2,f

   movlw d'7'
   movwf LOOPCOUNT

LOOPS3216C:
   rlf AARGB2,W
   rlf REMB1,f
   rlf REMB0,f
   movfw BARGB1
   btfss AARGB2,LSB
   gotobra SADD26LC

   subwf REMB1,f
   movfw BARGB0
   btfss _C
   incfsz BARGB0,W
   subwf REMB0,f
   gotobra SOK26LC

SADD26LC:
   addwf REMB1,f
   movfw BARGB0
   btfsc _C
   incfsz BARGB0,W
   addwf REMB0,f

SOK26LC:
   rlf AARGB2,f

   decfsz LOOPCOUNT,f
   gotobra LOOPS3216C

   rlf AARGB3,W
   rlf REMB1,f
   rlf REMB0,f
   movfw BARGB1
   btfss AARGB2,LSB
   gotobra SADD26L24

   subwf REMB1,f
   movfw BARGB0
   btfss _C
   incfsz BARGB0,W
   subwf REMB0,f
   gotobra SOK26L24

SADD26L24:
   addwf REMB1,f
   movfw BARGB0
   btfsc _C
   incfsz BARGB0,W
   addwf REMB0,f

SOK26L24:
   rlf AARGB3,f
   movlw d'7'
   movwf LOOPCOUNT

LOOPS3216D:
   rlf AARGB3,W
   rlf REMB1,f
   rlf REMB0,f
   movfw BARGB1
   btfss AARGB3,LSB
   gotobra SADD26LD

   subwf REMB1,f
   movfw BARGB0
   btfss _C
   incfsz BARGB0,W
   subwf REMB0,f
   gotobra SOK26LD

SADD26LD:
   addwf REMB1,f
   movfw BARGB0
   btfsc _C
   incfsz BARGB0,W
   addwf REMB0,f

SOK26LD:
   rlf AARGB3,f

   decfsz LOOPCOUNT,f
   gotobra LOOPS3216D

   btfsc AARGB3,LSB
   gotobra SOK26L
   movfw BARGB1
   addwf REMB1,f
   movfw BARGB0
   btfsc _C
   incfsz BARGB0,W
   addwf REMB0,f
SOK26L:
   endm

;************************************************************
;  FXD3216S:
;       32/16 Bit Signed Fixed Point Divide 32/16 -> 32.16
;
;       Input:  32 bit fixed point dividend in <AARGB0:AARGB1:AARGB2:AARGB3>
;               16 bit fixed point divisor in <BARGB0:BARGB1>
;
;       Use:    callps  FXD3216S
;
;       Output: 32 bit fixed point quotient in <AARGB0:AARGB1:AARGB2:AARGB3>
;               16 bit fixed point remainder in <REMB0:REMB1>
;
;       Result: AARG, REM  <--  AARG / BARG
;
;       Max Timing:     26+537+5 = 568 clks             A > 0, B > 0
;                       30+537+22 = 589 clks            A > 0, B < 0
;                       36+537+22 = 595 clks            A < 0, B > 0
;                       40+537+5 = 582 clks             A < 0, B < 0
;                                   10 clks             A = 0
;
;       Min Timing:     26+501+5 = 532 clks             A > 0, B > 0
;                       30+501+22 = 553 clks            A > 0, B < 0
;                       36+501+22 = 559 clks            A < 0, B > 0
;                       40+501+5 = 546 clks             A < 0, B < 0
;
;       PM: 40+157+21+55 = 273             DM: 12

FXD3216S:
   clrf SIGN
   clrf REMB0                          ; clear partial remainder
   clrf REMB1
   movfw AARGB0
   iorwf AARGB1,W
   iorwf AARGB2,W
   iorwf AARGB3,W
   btfsc _Z
   retlw h'00'

   movfw AARGB0
   xorwf BARGB0,W
   movwf TEMP
   btfsc TEMP,MSB
   comf SIGN,f

   clrf TEMPB3                         ; clear exception flag

   btfss BARGB0,MSB                    ; if MSB set, negate BARG
   gotobra CA3216S

   comf BARGB1,f
   comf BARGB0,f
   incf BARGB1,f
   btfsc _Z
   incf BARGB0,f

CA3216S:
   btfss AARGB0,MSB                    ; if MSB set, negate AARG
   gotobra C3216SX

   comf AARGB3,f
   comf AARGB2,f
   comf AARGB1,f
   comf AARGB0,f
   incf AARGB3,f
   btfsc _Z
   incf AARGB2,f
   btfsc _Z
   incf AARGB1,f
   btfsc _Z
   incf AARGB0,f

C3216SX:
   movfw AARGB0
   iorwf BARGB0,W
   movwf TEMP
   btfsc TEMP,MSB

   gotobra C3216SX1

C3216S:
   SDIV3216L

   btfsc TEMPB3,LSB                    ; test exception flag
   gotobra C3216SX4

C3216SOK:
   btfss SIGN,MSB
   retlw h'00'

   comf AARGB3,f
   comf AARGB2,f
   comf AARGB1,f
   comf AARGB0,f
   incf AARGB3,f
   btfsc _Z
   incf AARGB2,f
   btfsc _Z
   incf AARGB1,f
   btfsc _Z
   incf AARGB0,f

   comf REMB1,f
   comf REMB0,f
   incf REMB1,f
   btfsc _Z
   incf REMB0,f

   retlw h'00'

C3216SX1:
   btfss BARGB0,MSB                    ; test BARG exception
   gotobra C3216SX3
   btfsc AARGB0,MSB                    ; test AARG exception
   gotobra C3216SX2
   movfw AARGB2
   movwf REMB0
   movfw AARGB3
   movwf REMB1
   bcf REMB0,MSB
   rlf AARGB2,f
   rlf AARGB1,f
   rlf AARGB0,f
   movfw AARGB0
   movwf AARGB2
   movfw AARGB1
   movwf AARGB3
   clrf AARGB0
   clrf AARGB1
   gotobra C3216SOK
C3216SX2:
   clrf AARGB3                         ; quotient = 1, remainder = 0
   incf AARGB3,f
   clrf AARGB2
   clrf AARGB1
   clrf AARGB0
   retlw h'00'

C3216SX3:
   comf AARGB0,f                       ; numerator = h'07'FFFFFFF + 1
   comf AARGB1,f
   comf AARGB2,f
   comf AARGB3,f
   incf TEMPB3,f
   gotobra C3216S

C3216SX4:
   incf REMB1,f                        ; increment remainder and test for
   btfsc _Z
   incf REMB0,f
   movfw BARGB1                        ; overflow
   subwf REMB1,W
   btfss _Z
   gotobra C3216SOK
   movfw BARGB0                       ; overflow
   subwf REMB0,W
   btfss _Z
   gotobra C3216SOK
   clrf REMB0                          ; if remainder overflow, clear
   clrf REMB1
   incf AARGB3,f                       ; remainder, increment quotient and
   btfsc _Z
   incf AARGB2,f
   btfsc _Z
   incf AARGB1,f                       ; test for overflow exception
   btfsc _Z
   incf AARGB0,f
   btfss AARGB0,MSB
   gotobra C3216SOK
   bsf FPFLAGS,NAN
   retlw h'FF'

;************************************************************
;  displayTemperatureUnit - writes the temperature unit on the display

displayTemperatureUnit:                ; function displayTemperatureUnit() {
   displayCharLiteral degreeSign       ;    displayChar(degreeSign);
   movlw "C"                           ;    if (degreeC) {  // if 1
   btfss degreeC                       ;       displayChar("C");
   movlw "F"                           ;    } else {
   gotops displayChar                  ;       displayChar("F");  // Includes return
                                       ;    }  // if 1
;  return                              ; }  // displayTemperatureUnit()

;************************************************************
;  displayDecimalPoint - writes a decimal point to the display

displayDecimalPoint:
   movlw "."
   gotops displayChar                  ; displayChar includes return

;************************************************************
;  DSerror - no presence detect pulse from DS18B20 upon reset, so complain

DSerror:
   callps LCDhome                      ; Home LCD cursor
   movlw noSensorStringOffset          ; Display "Err" on the LCD
   callps displayMessage
   callps clearToEndOfLine             ; Clear anything else that was on that line
DSloop:
   gotobra DSloop

;************************************************************
;  readDS
;
;  Read the DS18B20's temperature registers into <currentDS18B20Temp+1:currentDS18B20Temp>,
;
;  The temperature data is in the first two bytes of the DS18B20's
;  scratchpad and the rest of the scratchpad is ignored (we reset
;  the DS18B20 after reading the temperature data)

readDS:
   callps OW_RESET                     ; Reset the DS18B20

   btfss PDBYTE                        ; Did we get a presence detect pulse?
   gotobra DSerror                     ; No: complain

   movlw SKIPROM                       ; issue 'skip ROM' command
   callps DSTXBYTE

   movlw READSCRATCHPAD                ; issue 'read scratchpad' command
   callps DSTXBYTE

   callps DSRXBYTE                     ; Read the scratchpad into currentDS18B20Temp
   movwf currentDS18B20Temp
   callps DSRXBYTE
   movwf currentDS18B20Temp+1

   gotops OW_RESET                     ; Stop DS18B20 from trying to send any more data

;  return

;************************************************************
;  convertTemperatureToBCD() converts temperature in <AARGB0:AARGB1> in 1/16 degree Celsius units into a BCD number
;     in bcd.
;
;     Inputs:
;        degreeC - if true, the conversion is to degrees C, otherwise conversion is to degrees F.
;        hysteresisTemperature - if true, conversion to degrees F is without the 32 degree offset
;        addRounding - if true, an amount is added to account for rounding, if false no rounding amount is added
;        addPoint5 - if addRounding and addPoint5 are true, 500 is added to the result (equivalent to adding 0.5 degrees for rounding to nearest integer)
;                    if addRounding is true but addPoint5 is false, 50 is added to the result (equivalent to adding 0.05 degrees for rounding to 1 decimal place)
;
;    Outputs:
;        bcd contains a BCD version of the input temperature, with rounding as specified, ready to be truncated and decimal point inserted for display.

convertTemperatureToBCD:               ; function convertTemperatureToBCD() {
   btfsc degreeC                       ;    if (!degreeC) {  // if 1
   gotobra CTTBif1else
                                       ;       // Compute temperature in degrees F times 10,000
                                       ;       // in <AARGB0:AARGB1:AARGB2:AARGB3> (32 bits)
                                       ;
                                       ;       // If hysteresis temperature:
                                       ;       // <AARGB0:AARGB1:AARGB2:AARGB3>
                                       ;       // = (<AARGB0:AARGB1> / 16 * 1.8) * 10,000;
                                       ;       // = <AARGB0:AARGB1> * 1125;
                                       ;
                                       ;       // If not hysteresis temperature:
                                       ;       // <AARGB0:AARGB1:AARGB2:AARGB3>
                                       ;       // = (<AARGB0:AARGB1> / 16 * 1.8 + 32) * 10,000;
                                       ;       // = <AARGB0:AARGB1> * 1125 + 320,000;

                                       ;       if (hysteresisTemperature) {
                                       ;          <AARGB0:AARGB1:AARGB2:AARGB3> = <AARGB0:AARGB1> * 1125;
                                       ;       } else {
                                       ;          <AARGB0:AARGB1:AARGB2:AARGB3> = <AARGB0:AARGB1> * 1125 + 320,000;
                                       ;       }

   copyL16 d'1125',BARGB1
   callps FXM1616S
   btfsc hysteresisTemperature
   gotobra CTTBnoOffset
   copyL32 d'320000',TEMPB3            ;
   add32 AARGB3,TEMPB3
CTTBnoOffset:
   gotobra CTTBif1end
CTTBif1else:                           ;    } else {
   copyL16 d'625',BARGB1               ;       <AARGB0:AARGB1:AARGB2:AARGB3> = <AARGB0:AARGB1> * 625;
   callps FXM1616S
                                       ;          // Temperature in degrees C times 10,000
                                       ;          // in <AARGB0:AARGB1:AARGB2:AARGB3> (32 bits)
CTTBif1end:                            ;    } // end if 1

   bcf displayNegative                 ;    displayNegative = false;
   btfss AARGB0,7                      ;    if (<AARGB0:AARGB1:AARGB2:AARGB3> < 0) {  // if 2
   gotobra CTTBif2end
   neg32 AARGB3                        ;       <AARGB0:AARGB1:AARGB2:AARGB3> = -<AARGB0:AARGB1:AARGB2:AARGB3>;
   bsf displayNegative                 ;       displayNegative = true;
CTTBif2end:                            ;    }  // end if 2

   btfss addRounding                   ;    if (addRounding) {  // if 3
   gotobra CTTBif3end
   btfss addPoint5                     ;       if (addPoint5) {  // if 4
   gotobra CTTBif4else
   copyL32 d'5000',TEMPB3              ;          <AARGB0:AARGB1:AARGB2:AARGB3> += 5000;
   gotobra CTTBif4end
CTTBif4else:                           ;       } else {
   copyL32 d'500',TEMPB3               ;          <AARGB0:AARGB1:AARGB2:AARGB3> += 500;
CTTBif4end:                            ;       }  // end if 4
   add32 AARGB3,TEMPB3
CTTBif3end:                            ;    }  // end if 3
   gotops binaryToBCD                  ;    binaryToBCD();   // Includes return
                                       ; }  // convertTemperatureToBCD()

;************************************************************
;  displayTemperature - displays a 16-bit temperature sensor reading in <AARGB0:AARGB1> on the LCD
;
;     The value in <AARGB0:AARGB1> is in 1/16 degree C units.
;     If degreeC is set, the value is displayed as degrees C.
;     If degreeC is clear, the value is displayed as degrees F.
;
;     If oneDecimalPlace is set, temperatures below 100 degrees (C or F) are displayed as 1 or 2 digits followed
;     by a decimal point and a final digit. Temperatures above 100 degrees are rounded to the nearest whole number of degrees
;     and displayed with no decimal place.
;
;     If oneDecimalPlace is not set, temperatures are rounded to the nearest number
;     of whole degrees (C or F) for display with no decimal places.
;
;  Algorithm: it is messy to do arithmetic with BCD values, so we convert to BCD first without attempting to do any rounding,
;     then look at how many digits there are and whether the oneDecimalPlace flag has been set to determine whether to round for
;     0 or 1 digits following the decimal point. Then the BCD conversion is repeated, but this time adding the appropriate
;     rounding amount to the binary number before binaryToBCD() is called.
;
; Note: if the temperature is between 99.95 and 100 degrees, it will be treated here as if it should be displayed with 1 decimal place
; and 0.05 will be added to it. However, this will cause it to become 100 degrees or more, and displayBCD will therefore truncate it to a whole number.
; For example, 99.959 -> 100.009 -> 100. This behaviour is still correct.

displayTemperature:                    ; function displayTemperature() {
   copy16 AARGB1,temperatureTemp       ;    temperatureTemp = <AARGB0,AARGB1>
   bcf addPoint5                       ;    addPoint5 = false;
   bcf addRounding                     ;    addRounding = false;
   callps convertTemperatureToBCD      ;    convertTemperatureToBCD();

   btfss oneDecimalPlace               ;    if (!oneDecimalPlace) {
   bsf addPoint5                       ;       addPoint5 = true;
                                       ;    }

   swapf  bcd+1,W                      ;    if (digit3 != 0) {
   andlw  h'0F'
   btfss  _Z
   bsf addPoint5                       ;       addPoint5 = true;
                                       ;    }

   movfw bcd+1                         ;    if (digit4 != 0) {
   andlw h'0F'
   btfss _Z
   bsf addPoint5                       ;       addPoint5 = true;
                                       ;    }

   bsf addRounding                     ;    addRounding = true;
   copy16 temperatureTemp,AARGB1       ;    <AARGB0,AARGB1> = temperatureTemp;
   callps convertTemperatureToBCD      ;    convertTemperatureToBCD();
   btfss displayNegative               ;    if (displayNegative) {
   gotobra DTif4end
   displayCharLiteral "-"              ;       displayChar("-");
DTif4end:                              ;    }  // end if 4
                                       ;       // Convert 32-bit binary in <AARGB0:AARGB1:AARGB2:AARGB3> to
                                       ;       // BCD in <bcd+0:bcd+1:bcd+2:bcd+3:bcd+4>
;  callps displayBCD                   ;    displayBCD();   // Fall through
                                       ;       // Display it with leading zeroes suppressed and the
                                       ;       // decimal point in the correct place
;  return                              ; }  // displayTemperature()

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
;  displayBCD - displays a BCD value in bcd+0 : bcd+1 : bcd+2 : bcd+3 : bcd+4 (bcd+0 high nibble is digit 1, and so on).
;               bcd+0 and the high nibble of bcd+1 are assumed to be zero. The decimal point is fixed between digits 6 and 7.
;               Leading zeroes are suppressed in the display. If oneDecimalPlace is true, a decimal point and digit 7 are shown if the integer
;               part of the number is less than 3 digits long. Whether or not a decimal point is shown, the number is properly rounded for display.
;
;  This subroutine is used to display the temperature reading
;
;Digit: 123456 7890
;       001203.4567
;BCD    0 1 2  3 4

displayBCD:

;------------------------------------------------------------
digit3:
   swapf  bcd+1,W
   andlw  h'0F'
   btfsc  _Z
   gotobra digit4
doDigit3:
   callps display_bcd1_left
   gotobra doDigit4                    ; Mandatory display of digit 4 if digit 3 displayed

;------------------------------------------------------------
digit4:
   movfw bcd+1
   andlw h'0F'
   btfsc _Z
   gotobra digit5
doDigit4:
   callps display_bcd1_right
   bcf oneDecimalPlace                 ; Don't show the decimal point or the digit after it if the
                                       ;   integer part of the number being displayed is 3 or 4 digits long
   gotobra doDigit5                    ; Mandatory display of digit 5 if digit 4 displayed
;------------------------------------------------------------
digit5:
   swapf  bcd+2,W
   andlw  h'0F'
   btfsc  _Z
   gotobra digit6
doDigit5:
   callps display_bcd2_left

;------------------------------------------------------------
; Digit 6 is mandatory even if it and all preceding digits are zero, since it either
; precedes the decimal point or is the last digit

digit6:
   callps display_bcd2_right
   btfss oneDecimalPlace
   return
   callps displayDecimalPoint          ; Put decimal point after digit 6

;------------------------------------------------------------
digit7:
;  callps   display_bcd3_left          ; Fall through to display_bcd3_left
;  return                              ; Includes return

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
display_bcd3_left:
   swapf bcd+3,W
   gotops putNibble                    ; putNibble includes return

display_bcd3_right:
   movfw bcd+3
   gotops putNibble                    ; putNibble includes return

display_bcd1_left:
   swapf bcd+1,W
   gotops putNibble                    ; putNibble includes return

display_bcd1_right:
   movfw bcd+1
   gotops putNibble                    ; putNibble includes return

display_bcd2_left:
   swapf bcd+2,W
   gotops putNibble                    ; putNibble includes return

display_bcd2_right:
   movfw bcd+2
   gotops putNibble                    ; putNibble includes return

display_bcd4_right:
   movfw bcd+4
   gotops putNibble                    ; putNibble includes return

display_bcd4_left:
   swapf bcd+4,W
;  gotops putNibble                    ; Fall through to putNibble (includes return)

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
;  putNibble - sends a BCD nibble in the low 4 bits of W to LCD

putNibble:
   andlw h'0F'                         ; mask off other BCD digit
   addlw h'30'                         ; Convert binary to ASCII
;  callps displayChar                  ; Fall through to displayChar
;  return                              ; displayChar includes return

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
;  displayChar - sends data byte in W to the LCD display and adjusts count of characters on the current line

displayChar:                           ; function displayChar() {
   movf charsLeftOnLine,f              ;    if (charsLeftOnLine != 0) {  // if 1
   btfss _Z
   decf charsLeftOnLine,f              ;       charsLeftOnLine--;
;PLCDDBif2end:                         ;    }  // end if 1
   btfss display                       ;    if (display) {  // if 2
   return
   bsf RS                              ;       RS = true;         // Select LCD data register
;  callps putLCDByte                   ;       putLCDByte();      // Fall through to putLCDByte
;  return                              ;       // putLCDByte() includes return
                                       ;    }  // if 2
                                       ; }  // displayChar()

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
putLCDByte:
   movwf CHR                           ; Save character that we are going to display
   swapf CHR,W                         ; Swap upper and lower nibbles (LCD is in 4 bit mode)
   callps putLCDNibble                 ; Send upper nibble
   movfw CHR                           ; Get lower nibble
;  callps putLCDNibble                 ; Fall through to putLCDNibble to send lower nibble
;  return                              ; putLCDNibble()  // includes return

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
;  putLCDNibble - sends 4-bit nibble to LCD and waits
;
;     This subroutine only affects the PORTB bits connected to the LCD; other PORTB bits are
;     retained at the same value they were when this subroutine was called.

putLCDNibble:
   movwf putNibbleTemp                 ; save nibble for bit testing
   bankselNonZero LATB
   copy8 LATB,putNibbleTempPORTB       ; copy current PORTB output latch values
   bank0
   bcf putNibbleTempPORTB,3
   bcf putNibbleTempPORTB,4
   bcf putNibbleTempPORTB,1
   bcf putNibbleTempPORTB,7

   btfsc putNibbleTemp,0
   bsf putNibbleTempPORTB,3

   btfsc putNibbleTemp,1
   bsf putNibbleTempPORTB,4

   btfsc putNibbleTemp,2
   bsf putNibbleTempPORTB,1

   btfsc putNibbleTemp,3
   bsf putNibbleTempPORTB,7

   copy8 putNibbleTempPORTB,PORTB

   waste2cycles                        ; Give LCD sufficient set-up time before taking ENA high
   waste2cycles
   waste2cycles
   bsf ENA                             ; now all 4 are set, so toggle ENA line high
   waste2cycles                        ; Minimum ENA high time is 450ns.
   waste2cycles
   waste2cycles
   bcf ENA                             ; Take ENA low again
;  callps D200us                       ; Hold data steady for 200us
;  return

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
;  D200us - 200us delay

D200us:
   usDelay d'200'                      ; Delay 200us
   return

;************************************************************
;  neg32showMinus -
;         deals with negative numbers: shows a minus sign on the display and
;         negates the 32-bit 2's complement binary number in <AARGB0:AARGB1:AARGB2:AARGB3>,
;         returning a 32-bit right-justified positive number in <AARGB0:AARGB1:AARGB2:AARGB3>.

neg32showMinus:
   neg32 AARGB1                        ; Negate the temperature reading in <AARGB0:AARGB1:AARGB2:AARGB3>
   movlw "-"                           ; Show a minus sign on the display
   gotops displayChar
;  return

;************************************************************
;  putLCDCommandByte: sends LCD command in W to LCD display module in 4 bit mode

putLCDCommandByte:                     ; function putLCDCommandByte() {
   btfss display                       ;    if (display) {  // if 1
   return
   bcf RS                              ;       select LCD command register
   gotops putLCDByte                   ;       putLCDByte();  // includes return
                                       ;    }  // if 1
                                       ; }  // putLCDCommandByte()

;************************************************************
;  Main program
;
;  Structure of the program:
;     initialise() sets up a timer that expires after 50ms. The main loop repeatedly calls fgTimeSlice(), which
;     does less than 50ms of work, then waits for the timer to expire and when it does it resets it and returns.

main:                                  ; function main() {
   callps initialise                   ;    initialise();

   callps resetBacklightTimer          ;    resetBacklightTimer();

                                       ;    // If the multifunction button is pressed very first thing after power up, restore the contrast to
                                       ;    // its default setting and turn on the backlight so that the display is guaranteed to be visible.
   btfsc multifunctionButton           ;    if (buttonPressed) {  // if 1
   gotobra Mif1end
   callps resetContrast                ;       resetContrast();
Mif1end:                               ;    }  // if 1

Mwhile1:                               ;    while (true) {  // while1
   callps fgTimeSlice                  ;       fgTimeSlice();
   btfsc multifunctionButton           ;       if (buttonPressed) {  // if 2
   gotobra Mif2end
                                       ;          backlightDutyCycle = backlightOnDutyCycle;   // Turn backlight fully on while we wait for button to be released
   bankselNonZero CCPR2L               ;          // CCPR2L is in bank 5
   copyL8 backlightOnDutyCycle,CCPR2L
   bank0
   callps fgTimeButtonPress            ;          fgTimeButtonPress();
   callps fgTimeSlice                  ;          fgTimeSlice();                               // Enforce 50ms minimum button up time for debouncing
   btfss longPress                     ;          if (longPress) {  // if 3                    // Long presses take effect even if the backlight was dim or off
   gotobra Mif3else
   callps resetBacklightTimer          ;             resetBacklightTimer();
   callps buttonPress                  ;             buttonPress();
   gotobra Mif3end
Mif3else:                              ;          } else {  // if 3                            // Short presses...
                                       ;             if (backlightTimer <= backlightDimTime) {  // if 4
   bankselNonZero backlightTimer       ;                // backlightTimer is in bank 4
   movfw backlightTimer
   sublw backlightDimTime
   btfss _C                            ;                // C=1 if backlightTimer <= backlightDimTime
   gotobra Mif4else
   callps resetBacklightTimer          ;                resetBacklightTimer();                 // ... turn the backlight fully on if it was dimmed or off
   gotobra Mif4end
Mif4else:                              ;             } else {  // if 4
   callps resetBacklightTimer          ;                resetBacklightTimer();
   callps buttonPress                  ;                buttonPress();                         // ... take effect if the backlight was already fully on
Mif4end:                               ;             }  // if 4
Mif3end:                               ;          }  // if 3
Mif2end:                               ;       }  // if 2
   gotobra Mwhile1                     ;    }  // while1
                                       ; }  // main()

;************************************************************
initialise:                            ; function initialise() {

   callps readSettingsFromEEPROM       ; readSettingsFromEPROM();
   bsf display                         ; display = true;

;  Initialise input & output ports, etc

   bankselNonZero ANSELA               ; ANSELA and ANSELB are in bank 3
   copyL8 b'00000000',ANSELA           ; All port A pins digital
   copyL8 b'00000000',ANSELB           ; All port B pins digital

   bankselNonZero WPUA                 ; WPUA and WPUB are in bank 4
   copyL8 b'00100000',WPUA             ; Enable weak pull-up on RA5 only
   copyL8 b'11111111',WPUB             ; Enable weak pull-ups on all port B pins

                                       ; Set up for 32MHz clock
                                       ; FOSC<2:0> = 100 to use internal oscillator block for the system clock (FOSC is in the Configuration Word)
                                       ; IRCF<3:0> = 1110 sets to the 8 MHz HFINTOSC selection
                                       ; SPLLEN = 1, set to enable the 4x PLL
                                       ; SCS<1:0> = 00 to allow the FOSC<2:0> bits in the Configuration Word to determine the clock source

   bankselNonZero OSCCON               ; OSCCON and OPTION_REG are in bank 1
   copyL8 1 << IRCF3 | 1 << IRCF2 | 1 << IRCF1 | 0 << IRCF0 | 0 << SCS1 | 0 << SCS0 | 1 << SPLLEN,OSCCON

   bcf OPTION_REG,NOT_WPUEN            ; Enable weak pull-ups on Port B and RA5

                                       ; Configure the CCP module for PWM operation.

                                       ; The PWM signal for control of the LCD contrast will appear at CCP4
                                       ; (RA4, pin 3 of the 16F1827 in 18 pin PDIP package)

                                       ; The PWM signal for the backlight will appear at CCP2
                                       ; (RB6, pin 12 of the 18 pin PDIP package)

   bankselNonZero CCPTMRS              ; CCPTMRS is in bank 5

                                       ; Select:
                                       ;    timer2 for use with CCP4 (contrast PWM through RA4), and
                                       ;    timer6 for use with CCP2 (backlight PWM through RB6)

   copyL8 0 << C4TSEL1 | 0 << C4TSEL0 | 1 << C2TSEL1 | 0 << C2TSEL0,CCPTMRS

   bankselNonZero APFCON0              ; APFCON0 is in bank 2

                                       ; Steer P2A function to RB6
   copyL8 0 << CCP2SEL,APFCON0

   ;-----------------------------------------------------------------
   ; Set up CCP4 to generate the contrast voltage by driving the CCP4 pin (RA4) using PWM.
   ; The associated timer is Timer2.

   bank0
   callps applyContrastSetting         ; applyContrastSetting();  // Loads the 8 MSBs of the contrast duty cycle into the CCPR4L register

   bankselNonZero CCP4CON              ; CCP4CON is in bank 6

                                       ; Load the two LSBs of the contrast duty cycle into DC4B<1:0>,
                                       ;   and set CCP4 to PWM mode.

   copyL8 0 << DC4B1 | 0 << DC4B0 | 1 << CCP4M3 | 1 << CCP4M2 | 0 << CCP4M1 | 0 << CCP4M0,CCP4CON

   bank0                               ; PIR2, T2CON, and PR2 are in bank 0
   bcf PIR2,TMR2IF                     ; Clear the TMR2IF interrupt flag bit of the PIR2 register
   copyL8 h'FF',PR2                    ; Load the PR2 register with the PWM period value

                                       ; Configure the T2CKPS bits of the T2CON register for
                                       ;   a Timer2 prescale of 1:16; enable Timer2 by setting the
                                       ;   TMR2ON bit of the T2CON register

   copyL8 1 << TMR2ON | 1 << T2CKPS1 | 0 << T2CKPS0,T2CON

   ;-----------------------------------------------------------------
   ; Set up ECCP2 to drive the backlight through the P2A pin (steered to RB6, pin 12 of the PDIP package) using PWM
   ; The associated timer is Timer6.

   bankselNonZero CCP2CON              ; CCP2CON and CCPR2L are in bank 5

                                       ; Load the two LSBs of the backlight PWM duty cycle into DC4B<1:0>,
                                       ;   and set ECCP2 to PWM mode with P2A as single-output and active-high.

   copyL8 0 << P1M1 | 0 << P1M0 | 0 << DC2B1 | 0 << DC2B0 | 1 << CCP2M3 | 1 << CCP2M2 | 0 << CCP2M1 | 0 << CCP2M0,CCP2CON

   copyL8 backlightOnDutyCycle,CCPR2L  ; Load the 8 MSBs of the backlight duty cycle into the CCPR2L register

   bank0                               ; PIR3 is in bank 0
   bcf PIR3,TMR6IF                     ; Clear the TMR6IF interrupt flag bit of the PIR3 register

   bankselNonZero PR6                  ; PR6 and T6CON are in bank 8
   copyL8 h'FF',PR6                    ; Load the PR6 register with the PWM period value

                                       ; Configure the T6CKPS bits of the T6CON register for
                                       ;   a Timer6 prescale of 1:16; enable Timer6 by setting the
                                       ;   TMR6ON bit of the T6CON register

   copyL8 1 << TMR6ON | 1 << T6CKPS1 | 0 << T6CKPS0,T6CON

   bankselNonZero TRISA                ; TRISA is in bank 1

                                       ; Set port A data directions
                                       ; 1 = input, 0 = output
                                       ; Set all unused port A pins to outputs to minimise power consumption

                                       ; RA0 - (digital output) unused
                                       ; RA1 - (digital output) LCD E (Ena)
                                       ; RA2 - (digital output) unused
                                       ; RA3 - (digital output) unused
                                       ; RA4 - (digital output) PWM output for LCD contrast control
                                       ; RA5 - (digital input) multifunction button
                                       ; RA6 - (digital input) DS18B20 data
                                       ; RA7 - (digital output) LCD RS

;           76543210
   copyL8 b'01100000',TRISA            ; 1 = input, 0 = output

                                       ; Set port B data directions
                                       ;
                                       ; RB0 - (digital output) unused
                                       ; RB1 - (digital output) LCD2 (DB6)
                                       ; RB2 - (digital output) thermostat output to relay driver
                                       ; RB3 - (digital output) LCD0 (DB4)
                                       ; RB4 - (digital output) LCD1 (DB5)
                                       ; RB5 - (digital output) unused
                                       ; RB6 - (digital output) programming clock / backlight control
                                       ; RB7 - (digital output) programming data / LCD3 (DB7)
;           76543210
   copyL8 b'00000000',TRISB            ; 1 = input, 0 = output

   bank0
   bank0indirect
   bsf convertInterrupt                ; convertInterrupt = true;

   callps LCDinit                      ; initialise the LCD module and clear the display

   clrf interruptCount

   ; Setup timer 1 to expire every 50ms

   ;   T1CKPS<1:0> = 11 to set prescaler for divide by 8
   ;   T1OSCEN = 0 to disable the LP oscillator.
   ;   TMR1CS<1:0> = 00 to use Fosc/4

   copyL8 1 << T1CKPS1 | 1 << T1CKPS0 | 0 << T1OSCEN | 0 << TMR1ON | 0 << TMR1CS1 | 0 << TMR1CS0,T1CON
   gotops resetTimer                   ;    resetTimer();    // Start timer1 going. Includes return
;  return                              ; }  // initialise()

;**************************************************************
time1:                                 ; function time1() {
                                       ;    // Waits until the timer1 period (50ms) expires, then resets timer1 and returns
t1while1:
   btfss PIR1,TMR1IF                   ;    while (!timer1) {  // while 1. PIR1 is in bank 0
   gotobra t1while1                    ;    }  // while 1
;  gotops resetTimer                   ;    resetTimer();   // Fall through. Includes return
                                       ; }  // time1()
;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
; Calculations for setting timer1 for 50ms intervals: system clock runs at 32MHz, timer1 runs at 1/32 of this speed due to
; use of Fosc/4 and 1:8 prescaler, so it increments every 1us. 50ms is 50,000us, so timer1 needs to count
; 50,000. The instructions that are executed to reset the timer1 count after it overflows each take 1/8 us.
; We make sure there are exactly 16 instructions and then load timer1 to count 49,998 before overflowing.
;
; Therefore need to load the timer with 65536 - 49998.

resetTimer:                            ; function resetTimer() {
   bcf T1CON,TMR1ON                    ;    Turn timer1 off. T1CON is in bank 0
   bcf PIR1,TMR1IF                     ;    Clear the flag
   movlw high (d'65536' - d'49998')    ;    Set timer1 to overflow in 50ms (compensate for the 16 instruction cycles
   movwf TMR1H                         ;       = 2us between timer overflowing and being restarted with the new value)
   movlw low (d'65536' - d'49998')
   movwf TMR1L
   waste2cycles
   waste2cycles
   waste2cycles
   nop
   bsf T1CON,TMR1ON                    ;    Turn timer1 on
   return                              ; }  // resetTimer()

;**************************************************************
buttonPress:                           ; function buttonPress() {
   btfss longPress                     ;    if (longPress) {  // if 1
   gotobra BPif1else
   btfss thermostatManual              ;       if (thermostatManual) {  // if 2
   gotobra BPif2else
   invert1 thermostatOutput            ;          thermostatOutput = !thermostatOutput;
   callps writeThermostatOutputToEEPROM;          writeThermostatOutputToEEPROM();
   callps displayThermostatOutputState ;          displayThermostatOutputState();
   gotobra BPif2end
BPif2else:                             ;       } else {  // if 2
   bsf thermostatManual                ;          thermostatManual = true;
   callps writeThermostatManualToEEPROM;          writeThermostatManualToEEPROM();
   callps writeThermostatOutputToEEPROM;          writeThermostatOutputToEEPROM();
   callps LCDhome                      ;          LCDhome();
   callps displayThermostatMode        ;          displayThermostatMode();
BPif2end:                              ;       }  // if 2

   callps line2                        ;       line2();
   callps clearToEndOfLine             ;       clearToEndOfLine();
   gotops waitUntilNotPressed          ;       waitUntilNotPressed();    // Includes return
                                       ;    } else {  // if 1
BPif1else:
   callps thermostatMenu               ;       thermostatMenu();     // Adjust thermostat temperature, hysteresis and polarity
   callps resetBacklightTimer
   callps contrastMenu                 ;       contrastMenu();       // Select LCD contrast
   callps resetBacklightTimer
   callps unitMenu                     ;       unitMenu();           // Select temperature units
   callps resetBacklightTimer
   return
                                       ;    }  // end if 1
                                       ; }  // buttonPress()

;************************************************************
resetBacklightTimer:                   ; function resetBacklightTimer() {
   bankselNonZero backlightTimer       ;    // backlightTimer is in bank 4
   copyL8 backlightOnTime,backlightTimer ;  backlightTimer = backlightOnTime;            // Reset timer
                                       ;    backlightDutyCycle = backlightOnDutyCycle;   // Restore backlight to full brightness
   bankselNonZero CCPR2L               ;    // CCPR2L is in bank 5
   copyL8 backlightOnDutyCycle,CCPR2L
   bank0
   return                              ; }  // resetBacklightTimer()

;************************************************************
;  fgTimeSlice() does a 50ms timeslice in the foreground,
;     i.e. the timeslice process is allowed to write to the display.
;     Once a second (once every 20 calls to mainSwitchboard) it updates the display with the main display.
;     Uses the interruptCount variable to track progress through the 20 call cycle.

fgTimeSlice:                           ; function fgTimeSlice() {
   bsf display                         ;    display = true;
   gotobra mainSwitchboard             ;    mainSwitchboard();   // Includes return
                                       ; }  // fgTimeSlice()

;************************************************************
;  bgTimeSlice() does a 50ms timeslice in the background,
;     i.e. the timeslice process is not allowed to write to the display.
;
;     Uses the interruptCount variable to track progress through the 20 call cycle.
;
;     Returns with display == false. The caller should be careful to set display after
;     the call, if required.

bgTimeSlice:                           ; function bgTimeSlice() {
   bcf display                         ;    display = false;
;  gotobra mainSwitchboard             ;    mainSwitchboard();  // Fall through. Includes return
                                       ; }  // bgTimeSlice()

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
mainSwitchboard:
   callps time1
   movfw interruptCount
   incf interruptCount,f
   brw

interruptTable:
   ; First reading takes a full second to allow DS18B20 conversion to complete

   gotobra doDSconvert                 ; Initiate DS18B20 conversion

   return                              ; Need to make sure DS18B20 has had enough time to perform its conversion
   return
   return
   return
   return

   return
   return
   return
   return
   return

   return
   return
   return
   return
   return

   return
   return
   return
   return
   return

   ; Second and subsequent times through

interruptTableLoop:
   return
   return
   return
   return
   return

   return
   return
   return
   return
   return

   return
   return
   return
   return
   return

   return
   return
   return
   return
   gotobra processAndDisplay

;************************************************************
doDSconvert:
   gotops convertDS                    ;  Includes return

;************************************************************
processAndDisplay:                     ; function processAndDisplay() {
   IFDEF autosleep
                                       ;    // Don't count seconds if we are waiting for the button to be released.
                                       ;    // The timer will already be set to 30 seconds and the backlight will be on full

   btfss multifunctionButton           ;    if (!buttonPress) {  // if 1
   gotobra PADif1end
   bankselNonZero backlightTimer       ;       // backlightTimer is in bank 4
                                       ;       if (backlightTimer != 0) {  // if 2
   movf backlightTimer,f
   btfsc _Z
   gotobra PADif2end
   decfsz backlightTimer,f             ;          backlightTimer--;
   gotobra PADif3else                  ;          if (backlightTimer == 0) {  // if 3
   bankselNonZero CCPR2L               ;             // CCPR2L is in bank 5
   copyL8 backlightOffDutyCycle,CCPR2L ;             backlightDutyCycle = backlightOffDutyCycle;
   gotobra PADif3end
PADif3else:                            ;          } else {  // if 3
                                       ;             if (backlightTimer == 10) {  // if 4
   compareL8skipEqual backlightTimer,backlightDimTime
   gotobra PADif4end
   bankselNonZero CCPR2L               ;                // CCPR2L is in bank 5
   copyL8 backlightDimDutyCycle,CCPR2L ;                backlightDutyCycle = backlightDimDutyCycle;
PADif4end:                             ;             }  // if 4
PADif3end:                             ;          }  // if 3
PADif2end:                             ;       }  // if 2
PADif1end:                             ;    }  // if 1
   bank0
   ENDIF ; autosleep
   callps readDS                       ;    currentDS18B20Temp = readDS();
   callps convertDS                    ;    convertDS();
                                       ;    interruptCount = interruptTableLoop - interruptTable;
   copyL8 (interruptTableLoop - interruptTable),interruptCount

   btfss thermostatManual              ;    if (thermostatManual) {  // if 6
   gotobra PADif6else
   callps LCDhome                      ;       LCDhome();
   movlw manualStringOffset            ;       displayMessage("Manual");
   callps displayMessage
   gotobra PADif6end
PADif6else:                            ;    else {
   callps displayThermostatSetting     ;       displayThermostatSetting();
PADif6end:                             ;    }  // if 6
   callps clearToEndOfLine             ;    clearToEndOfLine();
   callps line2                        ;    line2();
   copy16 currentDS18B20Temp,AARGB1    ;    <AARGB0:AARGB1> = currentDS18B20Temp;
   callps doThermostat                 ;    doThermostat();
   bsf oneDecimalPlace                 ;    oneDecimalPlace = true;
   btfsc display                       ;    if (display) {  // if 9
   bcf hysteresisTemperature           ;       hysteresisTemperature = false;
                                       ;    }  // if 9
   callps displayTemperature           ;    displayTemperature(currentDS18B20Temp);
   callps displayTemperatureUnit       ;    displayTemperatureUnit();
   displayCharLiteral " "              ;    displayChar(" ");
   callps displayThermostatState       ;    displayThermostatState();
   return                              ; }  // processAndDisplay()

;************************************************************
displayThermostatInvert:                ; function displayThermostatInvert() {
                                       ;    if (thermostatInvert) {  // if 1
                                       ;       displayMessage("ON");
                                       ;    else {
                                       ;       displayMessage("OFF");
                                       ;    }  // if 1
   movlw ONStringOffset
   btfss thermostatInvert
   movlw OFFStringOffset
   callps displayMessage
   gotops clearToEndOfLine             ;    clearToEndOfLine();     // Includes return
                                       ; }  // displayThermostatInvert()

;************************************************************
displayThermostatState:                ; function displayThermostatState() {
                                       ;    if (thermostatOutput) {  // if 1
                                       ;       displayMessage("ON");
                                       ;    else {
                                       ;       displayMessage("OFF");
                                       ;    }  // if 1
   movlw ONStringOffset
   btfss thermostatOutput
   movlw OFFStringOffset
   callps displayMessage
   gotops clearToEndOfLine             ;    clearToEndOfLine();     // Includes return
                                       ; }  // displayThermostatState()

;************************************************************
displayThermostatMode:                 ; function displayThermostatMode() {
                                       ;    if (thermostatManual) {  // if 1
                                       ;       displayMessage("MANUAL");
                                       ;    else {
                                       ;       displayMessage("AUTO");
                                       ;    }  // if 1
   movlw MANUALStringOffset
   btfss thermostatManual
   movlw AUTOStringOffset
   callps displayMessage
   gotops clearToEndOfLine             ;    clearToEndOfLine();     // Includes return
                                       ; }  // displayThermostatMode()

;************************************************************
displayThermostatSetting:              ; function displayThermostatSetting() {
   callps LCDhome                      ;    LCDhome();
                                       ;    if (thermostatInvert) {  // if 1
                                       ;       displayMessage("off");
                                       ;    } else {
                                       ;       displayMessage("on");
   movlw offStringOffset
   btfss thermostatInvert
   movlw onStringOffset
   callps displayMessage
                                       ;    }  // if 1
   displayCharLiteral ">"              ;    displayCharLiteral(">");
   copy16 thermostatTemperature,AARGB1 ;    displayTemperature(thermostatTemperature);
   bcf oneDecimalPlace
   btfsc display                       ;    if (display) {  // if 2
   bcf hysteresisTemperature           ;       hysteresisTemperature = false;
                                       ;    }  // if 2
   callps displayTemperature
   displayCharLiteral " "              ;    displayCharLiteral(" ");

                                       ;    if (thermostatInvert) {  // if 3
                                       ;       displayMessage("on");
                                       ;    } else {
                                       ;       displayMessage("off");
   movlw onStringOffset
   btfss thermostatInvert
   movlw offStringOffset
   callps displayMessage
                                       ;    }  // if 3
   displayCharLiteral "<"              ;    displayCharLiteral("<");
   btfsc display                       ;    if (display) {  // if 4
   bcf hysteresisTemperature           ;       hysteresisTemperature = false;
                                       ;    }  // if 4
   copy16 thermostatTemperature,AARGB1 ;    displayTemperature(thermostatTemperature - thermostatHysteresis);
   sub16 AARGB1,thermostatHysteresis
   bcf oneDecimalPlace
   gotops displayTemperature           ;    // Includes return
;  return                              ; }  // displayThermostatSetting()

;************************************************************
doThermostat:                          ; function doThermostat() {
                                       ;    // Precondition: <AARGB0:AARGB1> is a temperature in 1/16 degree C, 2s complement format.
                                       ;    // Postcondition: <AARGB0:AARGB1> is unchanged. The thermostatOutput is set or cleared according
                                       ;    //    to the thermostat settings.

   btfsc thermostatManual              ;    if (!thermostatManual) {  // if 1
   return
;  gotobra DTif1end
   copy16 AARGB1,TEMPB1                ;       if (<AARGB0:AARGB1> > thermostatTemperature) {  // if 2
   sub16 TEMPB1,thermostatTemperature
   btfsc TEMPB0,7
   gotobra DTif2else
   btfss thermostatInvert              ;          thermostatOutput = !thermostatInvert;
   bsf thermostatOutput
   btfsc thermostatInvert
   bcf thermostatOutput
   return
DTif2else:                             ;       } else {
   copy16 AARGB1,TEMPB1                ;          if (<AARGB0:AARGB1> < (thermostatTemperature - thermostatHysteresis) ) {  // if 3
   add16 TEMPB1,thermostatHysteresis
   sub16 TEMPB1,thermostatTemperature
   btfss TEMPB0,7
   gotobra DTif3end
   btfsc thermostatInvert              ;             thermostatOutput = thermostatInvert;
   bsf thermostatOutput
   btfss thermostatInvert
   bcf thermostatOutput
DTif3end:                              ;          }  // if 3
;DTif2end:                             ;       }  // if 2
;DTif1end:                             ;    }  // if 1
   return                              ; }  // doThermostat()

;************************************************************
unitMenu:                              ; function unitMenu() {
   bcf changedSetting                  ;    changedSetting = false;   // True iff we have to write back to EEPROM
UMdoWhile1:                            ;    do {
   callps LCDhome                      ;       LCDhome();
   movlw unitStringOffset              ;       displayMessage("Units: degrees ");
   callps displayMessage
   movlw "F"                           ;       if (degreeC) {
   btfsc degreeC                       ;          displayChar("C");
   movlw "C"                           ;       } else {
   callps displayChar                  ;          displayChar("F");
                                       ;       }
   callps clearToEndOfLine             ;       clearToEndOfLine();
   callps holdToChange                 ;       holdToChange();
   btfss longPress                     ;       if (longPress) {  // if 2
   gotobra UMdoWhile1end
   invert1 changedSetting              ;          changedSetting = !changedSetting;
   invert1 degreeC                     ;          degreeC = !degreeC;
   gotobra UMdoWhile1
                                       ;       }  // if 2
                                       ;    } while (longPress);
UMdoWhile1end:
   btfss changedSetting                ;    if (changedSetting) {  // if 2
   return
   copy16 thermostatTemperature,AARGB1 ;       thermostatTemperature = roundTemperature(thermostatTemperature);
   callps roundTemperature
   copy16 AARGB1,thermostatTemperature

                                       ;       EEPROM[thermostatTemperatureHEEPROMaddr] = thermostatTemperatureH;
   movlw thermostatTemperatureHEEPROMaddr-EEPROMstart
   bankselNonZero EEADR
   movwf EEADR
   banksel thermostatTemperatureH
   movfw thermostatTemperatureH
   callps writeEEPROM
                                       ;       EEPROM[thermostatTemperatureLEEPROMaddr] = thermostatTemperatureL;
   movlw thermostatTemperatureLEEPROMaddr-EEPROMstart
   bankselNonZero EEADR
   movwf EEADR
   banksel thermostatTemperatureL
   movfw thermostatTemperatureL
   callps writeEEPROM

   copy16 thermostatHysteresis,AARGB1  ;       thermostatHysteresis = roundTemperature(thermostatHysteresis);
   callps roundTemperature
   copy16 AARGB1,thermostatHysteresis

                                       ;       EEPROM[thermostatHysteresisHEEPROMaddr] = thermostatHysteresisH;
   movlw thermostatHysteresisHEEPROMaddr-EEPROMstart
   bankselNonZero EEADR
   movwf EEADR
   banksel thermostatHysteresisH
   movfw thermostatHysteresisH
   callps writeEEPROM
                                       ;       EEPROM[thermostatHysteresisLEEPROMaddr] = thermostatHysteresisL;
   movlw thermostatHysteresisLEEPROMaddr-EEPROMstart
   bankselNonZero EEADR
   movwf EEADR
   banksel thermostatHysteresisL
   movfw thermostatHysteresisL
   callps writeEEPROM
                                       ;       if (degreeC) {  // if 3
   movlw unitEEPROMaddr-EEPROMstart
   bankselNonZero EEADR
   movwf EEADR
   banksel flags1
   movlw "C"                           ;          EEPROM[unitEEPROMaddr] = "C";
   btfss degreeC                       ;       } else {
   movlw "F"                           ;          EEPROM[unitEEPROMaddr] = "F";
   gotops writeEEPROM                  ;       }  // end if 3
;  return                              ;    }  // end if 2
                                       ; }  // unitMenu()

;************************************************************
;  roundTemperature rounds a 1/16 degree C temperature value in <AARGB0:AARGB1> to the nearest 1 degree C if degreeC is true, or
;     to the nearest 1 degree F if degreeC is false.
;
;     For example, with degreeC true, -14 degrees C will be stored as -14 x 16 = -224.
;
;     If degreeC is then set to false and roundTemperature is called, -14 degrees C = 6.8 degrees F, which is rounded to 7 degrees F,
;     and the nearest 1/16 degree C value is -222.
;
;     If degreeC is then set to true and roundTemperature is called again, -222 / 16 = -13.875 degrees C, which is rounded to -14 degrees C,
;     represented by -224.

roundTemperature:                      ; function roundTemperature(<AARGB0:AARGB1>) {
                                       ;    var negative;
                                       ;
   btfss degreeC                       ;    if (degreeC) {  // if 1
   gotobra RTif1else
   gotops round16                      ;       return ( round16(C16temperature) );   // Includes return
;  gotobra RTif1end
RTif1else:                             ;    else {
                                       ;       // Degrees F
                                       ;
                                       ;       // F = C x 9/5 + 32
                                       ;       // F16 = (C16 / 16 * 9 / 5 + 32) * 16
                                       ;       //     = C16 * 9 / 5 + 32*16
                                       ;       //     = C16 * 9 / 5 + 512
                                       ;       // Round to nearest whole degree F:
                                       ;       //    F16 = F16 + 8;
                                       ;       //    F16 = trunc(F16 / 16) * 16;
                                       ;       // Convert back to a C16 number:
                                       ;       //    C16 = ( (F16/16 - 32) * 5 / 9 ) * 16
                                       ;       //        = (F16 - 512) * 5 / 9
                                       ;
   copyL16 d'9',BARGB1                 ;       <AARGB0:AARGB1> = round16( C16temperature * 9 / 5 );
   callps FXM1616S
   copyL16 d'5',BARGB1
   callps FXD3216S
   copy16 AARGB3,AARGB1                ;            // Convert 32 bit result to 16 bit in <AARGB0:AARGB1>
   callps round16                      ;            // <AARGB0:AARGB1> is whole-number Fahrenheit equivalent of input <AARGB0:AARGB1> C16 temperature,
                                       ;            //   without the 32 degree F offset
   copyL16 d'5',BARGB1                 ;       return ( <AARGB0:AARGB1> * 5 / 9 );
   callps FXM1616S
   copyL16 d'9',BARGB1
   callps FXD3216S                     ;
   copy16 AARGB3,AARGB1                ;       // convert result from 32 bits to 16 bits
;RTif1end:                             ;    }  // if 1
   return                              ; }  // roundTemperature()

;************************************************************
round16:                               ; function round16(<AARGB0:AARGB1>) {
                                       ;    // Returns <AARGB0:AARGB1> rounded to the nearest multiple of 16. Input and output are signed 16 bit numbers

   bcf negative                        ;    negative = false;
   btfss AARGB0,7                      ;    if (<AARGB0:AARGB1> < 0) {  // if 1
   gotobra R16if1end
   bsf negative                        ;       negative = true;

   neg16 AARGB1                        ;       <AARGB0:AARGB1> = -<AARGB0:AARGB1>;
R16if1end:                             ;    }  // if 1
   add16_L16 AARGB1,d'8'               ;    <AARGB0:AARGB1> += 8;  // This is to make the value round when it is truncated
   movlw h'F0'                         ;    <AARGB0:AARGB1> = trunc(<AARGB0:AARGB1> / 16) * 16;   // Zero the last 4 bits to truncate the value to the nearest integer
   andwf AARGB1,f
   btfss negative                      ;    if (negative) {  // if 2
   return
   neg16 AARGB1                        ;       <AARGB0:AARGB1> = -<AARGB0:AARGB1>;
                                       ;    }  // if 2
   return                              ; }  // round16()

;************************************************************
;  waitUntilNotPressed - waits until the multifunction button is not pressed, then to assist in switch
;                        debouncing waits a further 100ms before returning. bgTimeSlice() is called
;                        while waiting so that temperature readings will continue to be made and the
;                        thermostat output will continue to cycle while the menus are displayed.

waitUntilNotPressed:                   ; function waitUntilNotPressed() {
WUNPdoWhile1:                          ;    do {
   callps bgTimeSlice                  ;       bgTimeSlice();
   btfss multifunctionButton           ;    } while (buttonPress);
   gotobra WUNPdoWhile1
   callps bgTimeSlice                  ;    bgTimeSlice();   // Wait 50ms for debouncing
   bsf display                         ;    display = true;
   return                              ; }  // waitUntilNotPressed()

;************************************************************
resetContrast:                         ; function resetContrast() {
                                       ;    // Turn the backlight on and reset display contrast to highest setting since
                                       ;    // user was pressing multifunction button at power-up
   bsf changedSetting                  ;    changedSetting = true;
   copyL8 d'255',contrast              ;    contrast = 255;
   gotops contrastMenuChangedSetting   ;    contrastMenuChangedSetting();  // Includes return
;  return                              ; }  // resetContrast()

;************************************************************
applyContrastSetting:                  ; function applyContrastSetting() {
   movfw contrast                      ;    CCPR4L = contrast;
   bankselNonZero CCPR4L               ;    // CCPR4L is in bank 6
   movwf CCPR4L                        ;    // Load the 8 MSBs of the duty cycle into the CCPR4L register
   bank0
   return                              ; }  // applyContrastSetting()

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
displayInteger8:                       ; function displayInteger8(AARGB3) {
                                       ;    // Displays the 8 bit unsigned integer <AARGB3> with leading zeros suppressed,
                                       ;    // and clears to the end of the line.

   clrf AARGB2                         ;    <AARGB0:AARGB1:AARGB2:AARGB3> = <AARGB3>;
   clrf AARGB1
   clrf AARGB0
   callps binaryToBCD                  ;    bcd = binaryToBCD(<AARGB0:AARGB1:AARGB2:AARGB3>);
   movlw h'0F'                         ;    if (digit6 != 0) {  // if 1
   andwf bcd+3,W
   btfsc _Z
   gotobra DI8if1else
   callps display_bcd3_right           ;       displayDigit(digit6);
   gotobra DI8if2                      ;       displayDigit(digit7);
DI8if1else:                            ;    } else {  if 1
   movlw h'F0'                         ;       if (digit7 != 0) {  // if 2
   andwf bcd+4,W
   btfsc _Z
   gotobra DI8if2end
DI8if2:
   callps display_bcd4_left            ;          displayDigit(digit7);
DI8if2end:                             ;       }  // if 2
;DI8if1end:                            ;    }  // if 1
   callps display_bcd4_right           ;    displayDigit(digit8);
   gotops clearToEndOfLine             ;    clearToEndOfLine();     // Includes return
;  return                              ; }  // displayInteger8()

;************************************************************
waitForPress:                          ; function waitForPress() {
                                       ;    // Waits for the multifunction button to be newly pressed. This means
                                       ;    // first waiting until the button isn't pressed.
                                       ;
                                       ;    // The bgTimeSlice subroutine is called to implement the
                                       ;    // delays between polling the multifunction button, so that the normal device
                                       ;    // functions continue in the background while menus are displayed.
                                       ;
                                       ;    // Returns with longPress true if button held down for 1s
                                       ;    // or more, false otherwise. If button is held down for 1s or more
                                       ;    // this subroutine returns before the button is released.
                                       ;    // This allows the caller to update the display to
                                       ;    // show that the long button press has been recognised.

WFPdoWhile1:                           ;    do {                      // waitUntilNotPressed() has been inlined here to reduce subroutine call stack depth
   callps bgTimeSlice                  ;       bgTimeSlice();
   btfss multifunctionButton           ;    } while (buttonPress);  // do...while 1
   gotobra WFPdoWhile1
WFPdoWhile2:                           ;    do {
   callps bgTimeSlice                  ;       bgTimeSlice();
   btfsc multifunctionButton           ;    } while (!buttonPressed);  // do...while 2
   gotobra WFPdoWhile2
                                       ;    if (backlightTimer <= backlightDimTime) {  // if 1
   bankselNonZero backlightTimer       ;       // backlightTimer is in bank 4
   movfw backlightTimer
   sublw backlightDimTime
   btfss _C                            ;       // C=1 if backlightTimer <= 10
   gotobra WFPif1else
   callps resetBacklightTimer          ;       resetBacklightTimer();   // Turn backlight fully on

                                       ;       // Wait for another press now that the backlight is on
WFPdoWhile3:                           ;       do {                    // Wait for button to be released
   callps bgTimeSlice                  ;          bgTimeSlice();
   btfss multifunctionButton           ;       } while (buttonPress);  // do...while 3
   gotobra WFPdoWhile3
WFPdoWhile4:                           ;       do {                    // Wait for button to be pressed
   callps bgTimeSlice                  ;          bgTimeSlice();
   btfsc multifunctionButton           ;       } while (!buttonPressed);  // do...while 4
   gotobra WFPdoWhile4
   gotobra WFPif1end
WFPif1else:                            ;    } else {  // if 1
   callps resetBacklightTimer          ;       resetBacklightTimer();   // Make sure backlight is on full
WFPif1end:                             ;    }  // if 1
   callps bgTimeSlice                  ;    bgTimeSlice();           // 50ms debounce delay
;  callps timeButtonPress              ;    timeButtonPress();       // Fallthrough. Includes return
;  return                              ; }  // waitForPress()

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
timeButtonPress:                       ; function timeButtonPress() {
   bsf display                         ;    display = true;
   copyL8 d'20',downCount              ;    downCount = 20;
   bcf longPress                       ;    longPress = false;
TBPwhile1:                             ;    while (buttonPressed && downCount != 0) {  // while 1
   btfsc multifunctionButton
;  gotobra WFPwhile1end
   return
   callps bgTimeSlice                  ;       bgTimeSlice();   // 50ms delay
   bsf display                         ;       display = true;
   decfsz downCount,f                  ;       downCount--;
   gotobra TBPwhile1                   ;    }  // end while 1
   bsf longPress                       ;    if (downCount == 0) { longPress = true; }
;TBPwhile1end:
   return                              ; }  // timeButtonPress()

;************************************************************
fgTimeButtonPress:                     ; function fgTimeButtonPress() {
   copyL8 d'20',downCount              ;    downCount = 20;
   bcf longPress                       ;    longPress = false;
fgTBPwhile1:                           ;    while (buttonPressed && downCount != 0) {  // while 1
   btfsc multifunctionButton
;  gotobra WFPwhile1end
   return
   callps fgTimeSlice                  ;       fgTimeSlice();   // 50ms delay
   decfsz downCount,f                  ;       downCount--;
   gotobra fgTBPwhile1                 ;    }  // end while 1
   bsf longPress                       ;    if (downCount == 0) { longPress = true; }
;fgTBPwhile1end:
   return                              ; }  // fgTimeButtonPress()

;**************************************************************
;  writeEEPROM - writes the contents of W to the EEPROM address in EEADRL register
;
;   Usage:
;
;   bankselNonZero EEADRL              ; EEADRL is in bank 3
;   movlw DATA_EE_ADDR                 ; DATA_EE_ADDR is the EEPROM address to write to (8 bits wide)
;   movwf EEADRL                       ; Data Memory Address to write
;   movlw DATA_EE_DATA                 ; DATA_EE_DATA is the byte to write to the EEPROM
;   callps writeEEPROM                 ; Returns with bank 0 selected

writeEEPROM:
   bankselNonZero EEDATL               ; EEDATL is in bank 3
   movwf EEDATL                        ; Data value to write
   bcf EECON1,CFGS                     ; Deselect Configuration space
   bcf EECON1,EEPGD                    ; Point to DATA memory
   bsf EECON1,WREN                     ; Enable writes
   bcf INTCON,GIE                      ; Disable interrupts
   movlw h'55'
   movwf EECON2                        ; Write h'55'
   movlw h'AA'
   movwf EECON2                        ; Write h'AA'
   bsf EECON1,WR                       ; Set WR bit to begin write
   bcf EECON1,WREN                     ; Disable writes
WEloop:
   btfsc EECON1,WR                     ; Wait for write to complete
   gotobra WEloop
   bank0
   bsf INTCON,GIE                      ; Enable interrupts
   return

;************************************************************
;  LCDinit - initialises the LCD module, 4-bit interface mode

LCDinit:
   callps ms200                        ; Wait for LCD module hardware reset
   bcf RS                              ; Prime LCD to receive commands rather than data
   bcf ENA                             ; Take enable line low
   movlw h'03'                         ; Send init code the first time
   callps putLCDNibble
   callps ms100                        ; wait for LCD to catch up

   movlw h'03'                         ; send init code the second time
   callps putLCDNibble

   movlw h'03'                         ; and then a third time
   callps putLCDNibble

   movlw h'02'                         ; then send function code for 4 bits
   callps putLCDNibble

   movlw h'0C'                         ; and code for display on
   callps putLCDCommandByteShortDelay

   movlw h'28'                         ; and code for 2 lines, 5x7 dots
   callps putLCDCommandByteShortDelay

   movlw h'06'                         ; then code for entry mode
   callps putLCDCommandByteShortDelay

;  callps LCDclear                     ; Fall through to LCDclear. Includes return
;  return

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
;  LCDclear - clears the LCD and homes the cursor

LCDclear:
   copyL8 d'16',charsLeftOnLine        ; charsLeftOnLine = 16;
   movlw h'01'                         ; Clear display
   gotops putLCDCommandByteLongDelay   ; putLCDCommandByteLongDelay(clearLCD);  // Includes return
;  return

;************************************************************
;  LCDhome - homes the cursor on the LCD without erasing
;            anything

LCDhome:
   copyL8 d'16',charsLeftOnLine        ; charsLeftOnLine = 16;
   movlw h'02'                         ; home display's cursor
;  callps putLCDCommandByteLongDelay   ; putLCDCommandByteLongDelay(homeLCD)
;  return

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
;  putLCDCommandByteLongDelay - strobes the LCD with a longer (2ms) delay following, necessary when clearing the LCD

putLCDCommandByteLongDelay:
   callps putLCDCommandByte
;  gotops ms2

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
   gotops ms2

;************************************************************
;  delayLoop - returns after (delay1 * 256 + delay2 + 1) * 2 us (after including calling overhead).
;
;  Max delay using this subroutine is 65537 * 2 us = 131ms

delayLoopPlus2:
   gotobra $+1                         ; 2 cycles

delayLoop:                             ; 16 cycles (2us) per iteration through this loop
   nop                                 ; 1 cycle
   gotobra $+1
   gotobra $+1
   gotobra $+1
   movlw d'1'                          ; 1 cycle
   subwf delay2,f                      ; 1 cycle
   clrw
   subwfb delay1,f
   movfw delay1
   iorwf delay2,W
   btfss _Z
   gotobra delayLoop
   gotobra $+1
   gotobra $+1
   gotobra $+1
   return

;************************************************************
msDelay: macro ms
   copyL8 high (ms * d'500' - 1),delay1
   copyL8 low (ms * d'500' - 1),delay2
   gotobra delayLoop
   endm

;************************************************************
;  ms50, ms200, ms100: delay for about 50ms, 200ms or 100ms
;    respectively (untrimmed)

ms2:
   msDelay d'2'
ms200:
   callps ms100
ms100:
   msDelay d'100'

;************************************************************
;  clearToEndOfLine: clears to the end of the current line of the LCD

clearToEndOfLine:                      ; function clearToEndOfLine() {
CTEOLwhile1:                           ;    while (charsLeftOnLine != 0) {  // while 1
   movfw charsLeftOnLine
   btfsc _Z
;  gotobra CTEOLwhile1end
   return
   displayCharLiteral " "              ;       displayChar(" ");     // this updates charsLeftOnLine
   gotobra CTEOLwhile1                 ;    }  // end while 1
;CTEOLwhile1end:
;  return                              ; } // clearToEndOfLine()

;************************************************************
;  line2: moves display cursor to start of second line

line2:
   copyL8 d'16',charsLeftOnLine        ; charsLeftOnLine = 16;
   movlw h'C0'                         ; address for start of 2nd line of display
;  callps putLCDCommandByteShortDelay  ; putLCDCommandByteShortDelay(line2LCD);
;  return

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
putLCDCommandByteShortDelay:
   callps putLCDCommandByte
   gotops D200us

;************************************************************
;  Convert 32-bit binary number in <AARGB0,1,2,3> into BCD
;  in <bcd>. The value in <AARGB0,1,2,3> is destroyed.

binaryToBCD:                           ; function binaryToBCD() {

b2bcd:
   clrf bcd+0                          ;    for (var i = 0; i < 5; i ++) {  // for 1
   clrf bcd+1                          ;       bcd[i] = 0;
   clrf bcd+2                          ;    }  // end for 1
   clrf bcd+3
   clrf bcd+4
   bank0indirect                       ;    // FSR0H should point to bank 0
   copyL8 d'32',TEMPB0                 ;    for (var COUNT = 0; COUNT < 32; COUNT++) {  // for 2
b2bcd2:
   copyL8 bcd,FSR0L
   copyL8 d'5',TEMPB1                  ;       for (i = 0; i < 5; i++) {  // for 3
b2bcd3:
                                       ;          // About to multiply the BCD partial result by 2, but need to
                                       ;          // correct for overflow from one nibble to the next, which
                                       ;          // happens when the nibble is 5 or more before the multiply.
                                       ;          // e.g.  h'24' becomes h'48', but h'26' should become h'52'
                                       ;          // rather than h'4C'. If a nibble is 5 or more, need to
                                       ;          // carry a 1 into the next most significant nibble. This
                                       ;          // carry-in cannot itself cause a further overflow, since the
                                       ;          // nibble receiving it will have an even value (i.e. bit 0 = 0).
                                       ;
                                       ;          // The test for >=5 is most easily implemented in PIC code
                                       ;          // by adding 3 and testing if the high bit of the nibble is then
                                       ;          // 1 (in other words, if the nibble is >= 8 after adding 3).
                                       ;
                                       ;          // This has the added advantage that adding 3 and losing the
                                       ;          // overflow is exactly what is needed to convert the nibble
                                       ;          // from a binary number bigger than 9 to a BCD digit, since
                                       ;          // the overflow loses 16 from the low nibble and adds only
                                       ;          // 10 to the next higher nibble, so we need to add 6 to the
                                       ;          // low nibble and this is achieved by adding 3 and then
                                       ;          // multiplying the whole thing by 2.
   movlw h'33'
   addwf INDF0,f                       ;          bcd[i] = bcd[i] + 51;     // add 3 to both nibbles
   btfsc INDF0,3                       ;          if (bcd[i] % 16 < 8) {
   andlw h'F0'                         ;             bcd[i] = bcd[i] - 3;   // no overflow from low nibble
                                       ;          }
   btfsc INDF0,7                       ;          if (bcd[i] < 128) {
   andlw h'0F'                         ;             bcd[i] = bcd[i] - 48;  // no overflow from high nibble
                                       ;          }
   subwf INDF0,f
   incf FSR0L,f                        ;       }  // end for 3
   decfsz TEMPB1,f
   gotobra b2bcd3
   rlf AARGB3,f                        ;       carry = binaryInput - binaryInput % (2 ^ 31);
   rlf AARGB2,f                        ;       binaryInput = (binaryInput - carry) * 2;
   rlf AARGB1,f                        ;       carry = carry / (2 ^ 31);
   rlf AARGB0,f

   rlf bcd+4,f                         ;       for (var i = 0; i < 5; i++) {  // for 4
   rlf bcd+3,f                         ;          bcd[i] = bcd[i] * 2 + carry - bcd[i] % 128;
   rlf bcd+2,f                         ;          carry = ( bcd[i] - bcd[i] % (2 ^ 7) ) / (2 ^ 7);
   rlf bcd+1,f                         ;       }  // end for 4
   rlf bcd+0,f

   decfsz TEMPB0,f
   gotobra b2bcd2                      ;    }  // end for 2
   return                              ; }  // binaryToBCD()

;************************************************************
readEEPROM:
   bankselNonZero EEADR
   movwf EEADR

;  FALLTHROUGH
;VVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVVV
readEEPROMsameAddress:
   bsf EECON1,RD
   movfw EEDATL
   bank0
   return

;************************************************************
   org h'F000'

EEPROMstart:
thermostatInvertEEPROMaddr:
   de " "                              ; Default thermostat invert setting is " " (do not invert)
thermostatManualEEPROMaddr:
   de "M"                              ; Default thermostat manual setting is "M" (manual)
thermostatStateEEPROMaddr:
   de "0"                              ; Default thermostat manual state setting is "0" (thermostat output is off)
thermostatTemperatureHEEPROMaddr:
   de high (d'20' * d'16')             ; Default thermostat temperature is 20 degrees C
thermostatTemperatureLEEPROMaddr:
   de low (d'20' * d'16')              ; Default thermostat temperature is 20 degrees C
thermostatHysteresisHEEPROMaddr:
   de high (d'2' * d'16')              ; Default thermostat hysteresis is 2 degrees C
thermostatHysteresisLEEPROMaddr:
   de low (d'2' * d'16')               ; Default thermostat hysteresis is 2 degrees C
unitEEPROMaddr:
   de "C"                              ; Default temperature unit is degrees Celsius
contrastEEPROMaddr:
   de h'FF'                            ; Default contrast setting is 255 (maximum contrast)
;************************************************************
   END
